library ieee;
use ieee.math_real.all;
-- Filename: testbench_uart_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Sat Feb 22 18:39:29 2025
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of testbench_uart is
    constant c_period : time := 10 ns;

    constant c_divisor_width            : natural := 2;
    constant c_fraction_width           : natural := 5;

    constant c_data_width               : natural := 1;
    constant c_has_parity               : boolean := false;
    constant c_odd_parity               : boolean := false; -- When "true", then the number of ones together in databits and paritybit must be odd, else even.
    constant c_number_of_stopbits       : natural := 1; -- Allowed >=1
    constant c_number_of_sync_flipflops : natural := 1; -- Allowed >=0
    signal clk                  : std_logic := '0';
    signal clock_divisor        : unsigned(c_divisor_width-1 downto 0);
    signal data_receive         : std_logic_vector(c_data_width-1 downto 0);
    signal data_send            : std_logic_vector(c_data_width-1 downto 0);
    signal fraction_denominator : unsigned(c_fraction_width-1 downto 0);
    signal fraction_nominator   : unsigned(c_fraction_width-1 downto 0);
    signal parity_err           : std_logic;
    signal ready_receive        : std_logic;
    signal ready_receive_flag   : std_logic;
    signal ready_send           : std_logic;
    signal res                  : std_logic;
    signal run_simulation       : std_logic := '1';
    signal send                 : std_logic;
    signal tx                   : std_logic;
    component uart is
        generic (
            constant g_divisor_width        : natural := 8; -- allowed values: >= 2
            constant g_fraction_width       : natural := 3; -- allowed values: >= 2
            constant g_data_width           : natural := 8;
            constant g_has_parity           : boolean := true;
            constant g_odd_parity           : boolean := false;
            constant g_number_of_stopbits   : natural := 1;
            constant g_number_of_sync_flipflops : natural := 2
        );
        port (
            clk_i                  : in  std_logic;
            clock_divisor_i        : in  unsigned(g_divisor_width-1 downto 0);
            data_i                 : in  std_logic_vector(g_data_width-1 downto 0);
            fraction_denominator_i : in  unsigned(g_fraction_width-1 downto 0);
            fraction_nominator_i   : in  unsigned(g_fraction_width-1 downto 0);
            res_i                  : in  std_logic;
            rx_i                   : in  std_logic;
            send_i                 : in  std_logic;
            data_o                 : out std_logic_vector(g_data_width-1 downto 0);
            parity_err_o           : out std_logic;
            ready_receive_o        : out std_logic;
            ready_send_o           : out std_logic;
            tx_o                   : out std_logic
        );
    end component;
begin
    res <= '1', '0' after 1 ns;
    process
    begin
        while run_simulation='1' loop
            clk <= not clk;
            wait for c_period/2;
        end loop;
        wait;
    end process;
    -- 100 MHz/115.2 kHz = 868.0555
    -- 0.0555 = 1/18
    -- clock_divisor        <= to_unsigned(868, c_divisor_width);
    -- fraction_nominator   <= to_unsigned(  1, c_fraction_width);
    -- fraction_denominator <= to_unsigned( 18, c_fraction_width);
    
    -- clock_divisor        <= to_unsigned( 4, c_divisor_width);
    -- fraction_nominator   <= to_unsigned( 0, c_fraction_width);
    -- fraction_denominator <= to_unsigned( 0, c_fraction_width);
    
    clock_divisor        <= to_unsigned( 3, c_divisor_width);
    fraction_nominator   <= to_unsigned( 0, c_fraction_width);
    fraction_denominator <= to_unsigned( 0, c_fraction_width);
    
    process
    begin
        run_simulation <= '1';
        data_send <= (others => '0');
        send <= '0';
        wait for 100 * c_period;
        wait for 1.1 * c_period;
        for i in 0 to 2**c_data_width-1 loop
            send <= '1', '0' after c_period;
            data_send <= std_logic_vector(to_unsigned(i, c_data_width));
            wait until ready_send='1' and ready_receive_flag='1';
            assert data_send=data_receive report "Wrong data received" severity warning;
            wait for c_period/10;
            wait for c_period;
        end loop;
        run_simulation <= '0';
        wait;
    end process;
    uart_inst : uart
        generic map (
            g_divisor_width            => c_divisor_width, -- allowed values: >= 2
            g_fraction_width           => c_fraction_width, -- allowed values: >= 2
            g_data_width               => c_data_width,
            g_has_parity               => c_has_parity,
            g_odd_parity               => c_odd_parity,
            g_number_of_stopbits       => c_number_of_stopbits,
            g_number_of_sync_flipflops => c_number_of_sync_flipflops
        )
        port map (
            clk_i                  => clk,
            clock_divisor_i        => clock_divisor,
            data_i                 => data_send,
            fraction_denominator_i => fraction_denominator,
            fraction_nominator_i   => fraction_nominator,
            res_i                  => res,
            rx_i                   => tx,
            send_i                 => send,
            data_o                 => data_receive,
            parity_err_o           => parity_err,
            ready_receive_o        => ready_receive,
            ready_send_o           => ready_send,
            tx_o                   => tx
        );
    process(res, clk)
    begin
        if res='1' then
            ready_receive_flag <= '0';
        elsif rising_edge(clk) then
            if ready_receive='1' then
                ready_receive_flag <= '1';
            elsif send='1' then
                ready_receive_flag <= '0';
            end if;
        end if;
    end process;
    process
    begin
        report "flipflop_statistics for instance " & testbench_uart'path_name & " signal ready_receive_flag uses 1 flipflop.";
        wait;
    end process;
end architecture;
