-- Filename: testbench_square_root_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Sun Jan 12 09:54:19 2025
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
architecture struct of testbench_square_root is
    constant c_period                   : time    := 10 ns;
    constant c_radicand_width           : natural := 12;
    constant c_additional_root_bits     : natural := c_radicand_width/2+c_radicand_width mod 2;
    constant c_latency                  : natural := (c_radicand_width + (c_radicand_width mod 2))/2 + c_additional_root_bits;
    signal clk            : std_logic := '0';
    signal radicand       : unsigned(c_radicand_width-1 downto 0) := (others => '0');
    signal ready          : std_logic;
    signal res            : std_logic;
    signal root           : unsigned(c_radicand_width/2-1+c_additional_root_bits+c_radicand_width mod 2 downto 0);
    signal root_frac      : unsigned(c_additional_root_bits downto 0);
    signal root_int       : unsigned(c_radicand_width/2-1+c_additional_root_bits+(c_radicand_width mod 2) downto c_additional_root_bits);
    signal run_simulation : std_logic := '1';
    signal start          : std_logic;
    component square_root is
        generic (
            constant g_radicand_width       : natural := 27;
            constant g_additional_root_bits : natural := 0;
            constant g_latency              : natural :=  1  -- recommended values: 
        );
        port (
            clk_i      : in  std_logic;
            radicand_i : in  unsigned(g_radicand_width-1 downto 0);
            res_i      : in  std_logic;
            start_i    : in  std_logic;
            ready_o    : out std_logic;
            root_o     : out unsigned(g_radicand_width/2-1+g_additional_root_bits+g_radicand_width mod 2 downto 0)
        );
    end component;
begin
    res <= '1', '0' after 1 ns;
    process
    begin
        while run_simulation='1' loop
            clk <= not clk;
            wait for c_period/2;
        end loop;
        wait;
    end process;
    square_root_inst : square_root
        generic map (
            g_radicand_width        => c_radicand_width,       -- allowed values: >1
            g_additional_root_bits  => c_additional_root_bits, -- allowed values: all
            g_latency               => c_latency               -- allowed values: all
        )
        port map (
            clk_i      => clk,
            radicand_i => radicand,
            res_i      => res,
            start_i    => start,
            ready_o    => ready,
            root_o     => root
        );
    process
        variable root_real          : real;
        variable delta              : real;
        variable delta_max          : real := 0.0;
        variable delta_per_cent     : real;
        variable delta_per_cent_max : real := 0.0;
    begin
        run_simulation <= '1';
        radicand       <= (others => '0');
        start          <= '0';
        delta_max      := 0.0;
        wait for 1.1 * c_period;
        for i in 0 to 2**c_radicand_width-1 loop
            start <= '1', '0' after c_period;
            radicand <= to_unsigned(i, c_radicand_width);
            wait until ready='1';
            wait for c_period/10;
    
            root_real := 0.0;
            for b in root'range loop
                if root(b)='1' then
                    root_real := root_real + 2.0**(b-c_additional_root_bits);
                end if;
            end loop;
            delta := sqrt(real(i)) - root_real; -- delta is always positive.
            assert delta<2.0**(-c_additional_root_bits) report "delta out of legal range = " & real'image(delta) severity error;
            if abs(delta)>delta_max then
                delta_max := abs(delta);
            end if;
            delta_per_cent := delta/sqrt(real(i))*100.0;
            if abs(delta_per_cent)>delta_per_cent_max then
                delta_per_cent_max := abs(delta_per_cent);
            end if;
    
            assert true report LF &
                               "radicand              = " & integer'image(i) & ": "   & LF &
                               "reference             = " & real'image(sqrt(real(i))) & LF &
                               "root                  = " & real'image(root_real)     & LF &
                               "delta                 = " & real'image(delta)         & LF &
                               "delta_per_cent        = " & real'image(delta_per_cent)
                               severity warning;
    
            wait for c_period;
        end loop;
        report "delta_max          = " & real'image(delta_max);
        report "delta_per_cent_max = " & real'image(delta_per_cent_max);
        run_simulation <= '0';
        wait;
    end process;
    -- Split root into an integer and and an fractional part:
    root_int  <= root(root'high downto c_additional_root_bits);
    process(root)
    begin
        -- root_frac has 1 bit more than needed, for the case when c_additional_root_bits=0.
        -- root_frac(0)='0' in all cases:
        root_frac <= (others => '0');
        for i in c_additional_root_bits-1 downto 0 loop
            root_frac(i+1) <= root(i);
        end loop;
    end process;
    
end architecture;
