-- Filename: testbench_square_root_compare_to_cordic_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Sun Jan 12 11:54:05 2025
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
architecture struct of testbench_square_root_compare_to_cordic is
    constant c_period                     : time    := 10 ns;
    constant c_radicand_width             : natural := 12;
    constant c_additional_root_bits       : natural := c_radicand_width/2+c_radicand_width mod 2;
    constant c_latency_square_root        : natural := (c_radicand_width + (c_radicand_width mod 2))/2 + c_additional_root_bits;
    constant c_latency_cordic_square_root : natural := (c_radicand_width + (c_radicand_width mod 2))/2; 
    signal clk               : std_logic := '0';
    signal radicand          : unsigned(c_radicand_width-1 downto 0) := (others => '0');
    signal ready             : std_logic;
    signal ready_cordic      : std_logic;
    signal res               : std_logic;
    signal root              : unsigned(c_radicand_width/2-1+c_additional_root_bits+c_radicand_width mod 2 downto 0);
    signal root_cordic       : unsigned(c_radicand_width/2+c_radicand_width mod 2 downto 0);
    signal root_cordic_all   : unsigned(2*(c_radicand_width/2+c_radicand_width mod 2) downto 0);
    signal root_cordic_fract : unsigned(c_radicand_width/2-1+c_radicand_width mod 2 downto 0);
    signal root_frac         : unsigned(c_additional_root_bits downto 0);
    signal root_int          : unsigned(c_radicand_width/2-1+c_additional_root_bits+(c_radicand_width mod 2) downto c_additional_root_bits);
    signal run_simulation    : std_logic := '1';
    signal start             : std_logic;
    component square_root is
        generic (
            constant g_radicand_width       : natural := 27;
            constant g_additional_root_bits : natural := 0;
            constant g_latency              : natural :=  1  -- recommended values: 
        );
        port (
            clk_i      : in  std_logic;
            radicand_i : in  unsigned(g_radicand_width-1 downto 0);
            res_i      : in  std_logic;
            start_i    : in  std_logic;
            ready_o    : out std_logic;
            root_o     : out unsigned(g_radicand_width/2-1+g_additional_root_bits+g_radicand_width mod 2 downto 0)
        );
    end component;
    component cordic_square_root is
        generic (
            constant g_radicand_width           : natural := 27;
            constant g_improved_accuracy        : boolean := true;
            constant g_latency_prepare          : natural :=  1; -- allowed values    : 0..1 (values >1 are set to 1)
            constant g_latency_shift_radicand   : natural :=  1; -- allowed values    : 0..1 (values >1 are set to 1)
            constant g_latency_rotate_by_cordic : natural :=  1; -- recommended values: 0..(g_radicand_width + g_radicand_width mod 2)/2
            constant g_latency_shift_back       : natural :=  1  -- allowed values    : 0..1 (values >1 are set to 1)
        );
        port (
            clk_i               : in  std_logic;
            radicand_i          : in  unsigned(g_radicand_width-1 downto 0);
            res_i               : in  std_logic;
            start_i             : in  std_logic;
            ready_o             : out std_logic;
            square_root_fract_o : out unsigned(g_radicand_width/2-1+g_radicand_width mod 2 downto 0);
            square_root_o       : out unsigned(g_radicand_width/2+g_radicand_width mod 2 downto 0)
        );
    end component;
begin
    res <= '1', '0' after 1 ns;
    process
    begin
        while run_simulation='1' loop
            clk <= not clk;
            wait for c_period/2;
        end loop;
        wait;
    end process;
    square_root_inst : square_root
        generic map (
            g_radicand_width        => c_radicand_width,       -- allowed values: >1
            g_additional_root_bits  => c_additional_root_bits, -- allowed values: all
            g_latency               => c_latency_square_root   -- allowed values: all
        )
        port map (
            clk_i      => clk,
            radicand_i => radicand,
            res_i      => res,
            start_i    => start,
            ready_o    => ready,
            root_o     => root
        );
    process
        variable root_real          : real;
        variable delta              : real;
        variable delta_max          : real := 0.0;
        variable delta_per_cent     : real;
        variable delta_per_cent_max : real := 0.0;
        variable root_cordic_real          : real;
        variable delta_cordic              : real;
        variable delta_cordic_max          : real := 0.0;
        variable delta_cordic_per_cent     : real;
        variable delta_cordic_per_cent_max : real := 0.0;
        variable square_root_better : integer := 0;
        variable square_root_better_flag : boolean;
    begin
        run_simulation <= '1';
        radicand       <= (others => '0');
        start          <= '0';
        delta_max      := 0.0;
        wait for 1.1 * c_period;
        for i in 0 to 2**c_radicand_width-1 loop
            start <= '1', '0' after c_period;
            radicand <= to_unsigned(i, c_radicand_width);
            wait until ready='1';
            wait for c_period/10;
    
            root_real := 0.0;
            for b in root'range loop
                if root(b)='1' then
                    root_real := root_real + 2.0**(b-c_additional_root_bits);
                end if;
            end loop;
            delta := root_real - sqrt(real(i));
            if abs(delta)>delta_max then
                delta_max := abs(delta);
            end if;
            delta_per_cent := delta/sqrt(real(i))*100.0;
            if abs(delta_per_cent)>delta_per_cent_max then
                delta_per_cent_max := abs(delta_per_cent);
            end if;
    
            root_cordic_real := 0.0;
            for b in root_cordic_all'range loop
                if root_cordic_all(b)='1' then
                    root_cordic_real := root_cordic_real + 2.0**(b-(c_radicand_width/2+c_radicand_width mod 2));
                end if;
            end loop;
            delta_cordic := root_cordic_real - sqrt(real(i));
            if abs(delta_cordic)>delta_cordic_max then
                delta_cordic_max := abs(delta_cordic);
            end if;
            delta_cordic_per_cent := delta_cordic/sqrt(real(i))*100.0;
            if abs(delta_cordic_per_cent)>delta_cordic_per_cent_max then
                delta_cordic_per_cent_max := abs(delta_cordic_per_cent);
            end if;
    
            if abs(delta)<=abs(delta_cordic) then
                square_root_better_flag := true;
            else
                square_root_better_flag := false;
            end if;
            if not square_root_better_flag then
                if abs(abs(delta_cordic_per_cent)-abs(delta_per_cent))<0.5 then
                    square_root_better_flag := true;
                end if;
            end if;
            if square_root_better_flag=true then
                square_root_better := square_root_better + 1;
            end if;
    
            assert square_root_better_flag report LF &
                                              "radicand              = " & integer'image(i) & ": "                    & LF &
                                              "reference             = " & real'image(sqrt(real(i)))                  & LF &
                                              "root                  = " & real'image(root_real)                      & LF &
                                              "delta                 = " & real'image(delta)                          & LF &
                                              "delta_per_cent        = " & real'image(delta_per_cent)                 & LF &
                                              "root_cordic           = " & real'image(root_cordic_real)               & LF &
                                              "delta_cordic          = " & real'image(delta_cordic)                   & LF &
                                              "delta_cordic_per_cent = " & real'image(delta_cordic_per_cent)
                                              severity warning;
    
            wait for c_period;
        end loop;
        report "delta_max          = " & real'image(delta_max);
        report "delta_cordic_max   = " & real'image(delta_cordic_max);
        report "square_root_better = " & integer'image(square_root_better);
        run_simulation <= '0';
        wait;
    end process;
    -- Split root into an integer and and an fractional part:
    root_int  <= root(root'high downto c_additional_root_bits);
    process(root)
    begin
        -- root_frac has 1 bit more than needed, for the case when c_additional_root_bits=0.
        -- root_frac(0)='0' in all cases:
        root_frac <= (others => '0');
        for i in c_additional_root_bits-1 downto 0 loop
            root_frac(i+1) <= root(i);
        end loop;
    end process;
    
    cordic_square_root_inst : cordic_square_root
        generic map (
            g_radicand_width           => c_radicand_width,
            g_improved_accuracy        => false,
            g_latency_prepare          => 0, -- allowed values    : 0..1 (values >1 are set to 1)
            g_latency_shift_radicand   => 0, -- allowed values    : 0..1 (values >1 are set to 1)
            g_latency_rotate_by_cordic => c_latency_cordic_square_root, -- recommended values: 0..(g_radicand_width + g_radicand_width mod 2)/2
            g_latency_shift_back       => 0  -- allowed values    : 0..1 (values >1 are set to 1)
        )
        port map (
            clk_i               => clk,
            radicand_i          => radicand,
            res_i               => res,
            start_i             => start,
            ready_o             => ready_cordic,
            square_root_fract_o => root_cordic_fract,
            square_root_o       => root_cordic
        );
    root_cordic_all <= root_cordic & root_cordic_fract;
end architecture;
