-- Filename: rotate_e.vhd
-- Created by HDL-SCHEM-Editor at Tue Mar 12 13:17:36 2024
--
-- Functionality:
-- This modules rotates any incoming vector by the cordic algorithm.
-- The cordic algorithm rotates a vector by several micro-rotations with decreasing small angles.
--
-- When the vector-mode is selected, then the vector is rotated to the positive x-axis.
-- In this case the actual sign of the y-coordinate determines the direction of each micro-rotation.
-- At the end the outgoing y-coordinate is always 0 or near to 0.
-- The outgoing x-coordinate is always positive and identical to the length of the incoming vector.
-- The resulting rotation angle is identical to the angle, the incoming vector has had with the positive x-axis.
-- So the vector mode can be used to convert cartesian coordinates into polar coordinates.
--
-- When the rotation-mode is selected, then the vector is rotated by the given rotation angle.
-- In this case the rotation angle is decreased by the angle of each micro-rotation and the sign
-- of the remaining rotation angle determines the rotation direction.
--
-- Configuration by the generics:
-- The cordic algorithm needs an infinite number of micro-rotations to be exact.
-- But of course the number of micro-rotations are limited here and so the results cannot be 100% exact.
-- The number of micro-rotations the module will execute depends from the number of bits A which are used for angles and
-- from the number of bits C for coordinates.
-- The delta-angle which is used at the first micro-rotation is always 45 degrees.
-- At each following micro-rotation the delta-angle is approximately halved.
-- When the angles between 0 and 360 degree are represented by A bits, then the 45 degree angle has the value "00100..000".
-- As the angle is approximately halved at every micro-rotation, A-2 micro-rotations can be executed,
-- before the next delta-angle changes to 0.
-- At each micro-rotation the new x and y coordinates are calculated by adding or subtracting a delta-value to the previous coordinates.
-- These delta-values are determined by shifting the previous coordinates to the right by N bits,
-- where N is the number of micro-rotations which already took place.
-- The module rotate_by_cordic works with coordinates which have C+2 bits (including 1 sign bit), because:
-- One additional bit is needed for the rotation of a vector with maximum length to the x-axis, the x-coordinate is then increased by factor 1.4.
-- A second additional bit is needed because the Cordic-algorithm first ignores the cos(phi) factors of the Cordic-algorithm and therefore
-- the coordinates are increased by factor 1.6.
-- As at the first micro-rotation the previous coordinates are not shifted, C+1 micro-rotations can be executed,
-- before the delta-values are 0.
-- Therefore it makes sense to fulfill this equation:
-- C + 1 = A - 2 
-- C     = A - 3  (1)
-- When the equation (1) is not fulfilled, then the number of micro-roations is determined by the maximum of C+1 and A-2.
-- An example for an appropriate configuration of the generics is (with N=28 micro-rotations):
-- g_angle_width      = A = 30
-- g_coordinate_width = C = A - 3 = 27
-- When a short vector (small x and small y coordinate) is rotated, the calculation of the delta-values runs into a problem:
-- Because of the limited number of bits the coordinates are stored in, the right shift produces 0 as delta-value for both
-- coordinates long before all the micro-rotations are executed. So the sign of the y-coordinate cannot change any more and
-- all following micro-rotations turn into the same direction, which would cause inaccurate results.
-- To avoid this first the incoming x- and y-coordinates are multiplied by an identical factor by shifting them to their
-- biggest possible length. And at the end the resulting vector is shortened by the same factor.
--
-- The generics for the latency can be configured independently from the other generics.
-- When the latency is 0, then the sub-module works without any clock edge.
-- When the latency is 1, then the sub-module needs 1 clock edge to produce the results.
-- The smaller the latency is the more the sub-module has to calculate in a clock period,
-- which can result in difficulties at reaching timing closure in synthesis.
-- As in 1 clock period at least 1 micro-rotation is done, the slowest appropriate
-- configuration for the example above with N=28 micro-rotations is:
-- g_angle_width               = 30
-- g_coordinate_width          = 27
-- g_latency_lengthen_vector   =  1
-- g_latency_rotate_by_90      =  1
-- g_latency_rotate_by_cordic  = 28 (identical to N=28 micro-rotations, so 1 micro-rotation is done in 1 clock period)
-- g_latency_fix_cordic_length =  1
-- g_latency_shorten_vector    =  1
-- If you select g_latency_rotate_by_cordic in the example above to be bigger than 28,
-- then additional micro-rotations with delta-angle=0 are executed.
-- If you select g_latency_rotate_by_cordic in the example above to be smaller than 28,
-- then more than 1 micro-rotation per clock period has to be done.
-- If you set g_latency_rotate_by_cordic=27 for example, then 2 micro-rotations per clock period have to be done.
-- As the latency is 27 the total number of micro-rotations calculates to 2*27=54. But as
-- only 28 micro-rotations can be calculated (because of number of bits of angle and coordinates),
-- there are 54-28=26 micro-rotations with delta-angle=0 added after the first 28 micro-rotations.
-- The maximum values for the generics are:
-- g_coordinate_width         = 664
-- g_angle_width              = 667
-- g_latency_rotate_by_cordic = 664 Be aware that this value must be selected in a way that never more than total 675 micro-rotations are used.

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity rotate is
    generic (
        constant g_angle_width               : natural := 30; -- Allowed values    : g_angle_width <= 666
        constant g_coordinate_width          : natural := 27; -- Recommended value : g_angle_width-3
        constant g_latency_lengthen_vector   : natural := 1;  -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_latency_rotate_by_90      : natural := 1;  -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_latency_rotate_by_cordic  : natural := 28; -- Recommended values: 0..g_angle_width-2
        constant g_latency_fix_cordic_length : natural := 1;  -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_latency_shorten_vector    : natural := 1;  -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_improved_accuracy         : boolean := true
    );
    port (
        clk_i            : in  std_logic;
        res_i            : in  std_logic;
        rotation_angle_i : in  signed(g_angle_width-1 downto 0);
        start_i          : in  std_logic;
        vector_mode_i    : in  std_logic;
        x_coord_i        : in  signed(g_coordinate_width-1 downto 0);
        y_coord_i        : in  signed(g_coordinate_width-1 downto 0);
        ready_o          : out std_logic;
        rotation_angle_o : out signed(g_angle_width-1 downto 0);
        x_coord_o        : out signed(g_coordinate_width downto 0);
        y_coord_o        : out signed(g_coordinate_width downto 0)
    );
end entity rotate;
