-- Filename: rotate_by_cordic_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Oct 18 14:30:42 2024
library ieee;
use ieee.math_real.all;
architecture struct of rotate_by_cordic is
    function max (number1 : natural; number2 : natural) return natural is
    begin
        if number1>=number2 then
            return number1;
        end if;
        return number2;
    end function;
    constant c_number_of_iterations : natural := max(g_coordinate_width+1, -- Here in this module the coordinates have g_coordinate_width+1 bits (including 1 sign bit),
                                                                           -- so g_coordinate_width shifts can be done before the coordinate gets 0, but at the first
                                                                           -- micro-rotation no shift is done, so g_coordinate_width+1 micro-rotations are possible.
                                                     g_angle_width-2);     -- The biggest delta-angle is 45 degrees, an angle with g_angle_width bits represents -180..+180 degrees.

    function calculate_number_of_additional_bits return natural is
        variable number_of_iterations            : real;
        variable number_of_additional_angle_bits : real;
        variable number_of_additional_bits       : natural;
    begin
        if g_improved_accuracy=false then
            return 0;
        end if;
        number_of_iterations := real(c_number_of_iterations);
        number_of_additional_bits := natural(ceil(log2(number_of_iterations)));
        if number_of_additional_bits<2 then
            number_of_additional_bits := 2;
        end if;
        return number_of_additional_bits;
    end function;
    constant c_additional_angle_bits : natural := calculate_number_of_additional_bits;
    constant c_additional_coord_bits : natural := c_additional_angle_bits;

    type t_signed_angle  is array (natural range <>) of signed(g_angle_width-1+c_additional_angle_bits downto 0);
    type t_signed_coord  is array (natural range <>) of signed(g_coordinate_width+1+c_additional_coord_bits downto 0);

    function calculate_nr_of_iterations_per_period return natural is
        variable remainder             : natural;
        variable iterations_per_period : natural;
    begin
        if g_latency=0 or g_latency=1 then
            iterations_per_period := c_number_of_iterations;
        else
            remainder := c_number_of_iterations rem g_latency;
            if remainder=0 then
                iterations_per_period := c_number_of_iterations/g_latency;
            else
                iterations_per_period := (c_number_of_iterations - remainder + g_latency)/g_latency;
            end if;
        end if;
        return iterations_per_period;
    end function ;
    constant c_nr_of_iterations_per_period : natural := calculate_nr_of_iterations_per_period;

    function calculate_nr_of_iterations_all return natural is
    begin
        if g_latency=0 or g_latency=1 then
            return c_number_of_iterations;
        else
            return g_latency * c_nr_of_iterations_per_period;
        end if;
    end function;
    constant c_nr_of_iterations_all : natural  := calculate_nr_of_iterations_all;
    constant c_nr_of_periods        : positive := c_nr_of_iterations_all/c_nr_of_iterations_per_period;

    type t_natural_array is array (natural range <>) of natural range 0 to c_nr_of_iterations_all-1;
    type t_step          is array (c_nr_of_periods-1 downto 0) of t_natural_array(c_nr_of_iterations_per_period-1 downto 0);
    function calculate_step return t_step is
        variable v_step : t_step;
    begin
        for period in 0 to c_nr_of_periods-1 loop
            for iteration in 0 to c_nr_of_iterations_per_period-1 loop
                -- Shift the entries by 1 position, because generating steps is clocked:
                if period=0 then
                    v_step(c_nr_of_periods-1)(iteration) := c_nr_of_iterations_per_period*period + iteration;
                else
                    v_step(period-1)         (iteration) := c_nr_of_iterations_per_period*period + iteration;
                end if;
            end loop;
        end loop;
        return v_step;
    end function;
    constant c_step : t_step := calculate_step;
    signal counter            : natural range 0 to c_nr_of_periods-1;
    signal enable_reg         : std_logic;
    signal first_step         : std_logic;
    signal positive_rotation  : std_logic_vector(c_nr_of_iterations_per_period-1 downto 0);
    signal ready_steps        : std_logic;
    signal rotation_angle     : t_signed_angle(c_nr_of_iterations_per_period downto 0);
    signal rotation_angle_ext : signed(g_angle_width-1+c_additional_angle_bits downto 0);
    signal rotation_angle_reg : signed(g_angle_width-1+c_additional_angle_bits downto 0);
    signal steps              : t_natural_array(c_nr_of_iterations_per_period-1 downto 0);
    signal x_coord            : t_signed_coord(c_nr_of_iterations_per_period downto 0);
    signal x_coord_ext        : signed(g_coordinate_width+1+c_additional_coord_bits downto 0);
    signal x_coord_reg        : signed(g_coordinate_width+1+c_additional_coord_bits downto 0);
    signal y_coord            : t_signed_coord(c_nr_of_iterations_per_period downto 0);
    signal y_coord_ext        : signed(g_coordinate_width+1+c_additional_coord_bits downto 0);
    signal y_coord_reg        : signed(g_coordinate_width+1+c_additional_coord_bits downto 0);
    component rotate_by_cordic_step is
        generic (
            constant g_number_of_iterations     : natural := 32;
            constant g_number_of_iterations_all : natural := 32;
            constant g_angle_width              : natural := 32;
            constant g_coordinate_width         : natural := 32
        );
        port (
            positive_rotation_i : in  std_logic;
            rotation_angle_i    : in  signed(g_angle_width-1 downto 0);
            step_i              : in  natural range 0 to g_number_of_iterations_all-1;
            x_coord_i           : in  signed(g_coordinate_width+1 downto 0);
            y_coord_i           : in  signed(g_coordinate_width+1 downto 0);
            rotation_angle_o    : out signed(g_angle_width-1 downto 0);
            x_coord_o           : out signed(g_coordinate_width+1 downto 0);
            y_coord_o           : out signed(g_coordinate_width+1 downto 0)
        );
    end component;
    component rotate_control is
        generic (
            constant g_counter_max : natural := 16
        );
        port (
            clk_i          : in  std_logic;
            res_i          : in  std_logic;
            start_cordic_i : in  std_logic;
            counter_o      : out natural range 0 to g_counter_max;
            enable_reg_o   : out std_logic;
            first_step_o   : out std_logic;
            ready_steps_o  : out std_logic
        );
    end component;
begin
    -- When g_latency=0, then the signals connected to the outputs ready_steps_o and
    -- enable_reg_o of rotation_control will not be read.
    -- The signal period will have the correct value 0 in this case.
    rotate_control_inst : rotate_control
        generic map (
            g_counter_max => c_nr_of_periods-1
        )
        port map (
            clk_i          => clk_i,
            res_i          => res_i,
            start_cordic_i => start_cordic_i,
            counter_o      => counter,
            enable_reg_o   => enable_reg,
            first_step_o   => first_step,
            ready_steps_o  => ready_steps
        );
    -- rotate_by_cordic increases the number of bits of the incoming coordinates by 1.
    -- As rotate_by_90 already has already increased the number of bits by 1 (without using the new range),
    -- now 2 additional bits are available.
    -- Both bits are needed, because the module rotate_by_cordic_step ignores the cos(phi) factors, which increases
    -- the vector by factor 1.607. This is also neccessary, because rotating a 45 degree vector with maximum length
    -- to the x-axis will create a vector whose x-coordinate will be increased by factor squareroot(2)=1.4.
    -- Together both factors can increase the vector by factor 1.6*1.4=2.2:
    process(rotation_angle_i, x_coord_i, y_coord_i)
    begin
        rotation_angle_ext <= (others => '0');
        rotation_angle_ext(rotation_angle_ext'high downto rotation_angle_ext'high-g_angle_width+1) <= rotation_angle_i;
        x_coord_ext <= (others => '0');
        x_coord_ext(x_coord_ext'high downto x_coord_ext'high-g_coordinate_width-1) <= x_coord_i(g_coordinate_width) & x_coord_i;
        y_coord_ext <= (others => '0');
        y_coord_ext(y_coord_ext'high downto y_coord_ext'high-g_coordinate_width-1) <= y_coord_i(g_coordinate_width) & y_coord_i;
    end process;
    x_coord(0)        <= x_coord_ext        when first_step='1' or g_latency=0 or g_latency=1 else x_coord_reg;
    y_coord(0)        <= y_coord_ext        when first_step='1' or g_latency=0 or g_latency=1 else y_coord_reg;
    rotation_angle(0) <= rotation_angle_ext when first_step='1' or g_latency=0 or g_latency=1 else rotation_angle_reg;
    process (res_i, clk_i)
    begin
        if res_i='1' then
            steps <= c_step(c_nr_of_periods-1);
        elsif rising_edge(clk_i) then
            if enable_reg='1' then
                steps <= c_step(counter);
            end if;
        end if;
    end process;
    rotate_by_cordic_step_g: for i in 0 to c_nr_of_iterations_per_period-1 generate
        -- In vector-mode the incoming vector is rotated to the positive part of the x-axis,
        -- until the y-coordinate is 0.
        -- In this case the rotation direction is determined by the sign of the y-coordinate of
        -- the rotated vector.
        -- In rotation-mode the incoming vector is rotated until the incoming rotation angle
        -- is decreased to 0 by the delta angles of the Cordic algorithm.
        -- In this case the sign of the remaining rotation-angle determines
        -- the rotation direction:
        positive_rotation(i) <= y_coord(i)(y_coord(0)'high) when vector_mode_i='1' else
                                not rotation_angle(i)(g_angle_width-1+c_additional_angle_bits); -- rotation-mode
        rotate_by_cordic_step_inst : rotate_by_cordic_step
            generic map (
                g_number_of_iterations     => c_number_of_iterations,
                g_number_of_iterations_all => c_nr_of_iterations_all,
                g_angle_width              => g_angle_width+c_additional_angle_bits,
                g_coordinate_width         => g_coordinate_width+c_additional_coord_bits
            )
            port map (
                positive_rotation_i => positive_rotation(i),
                rotation_angle_i    => rotation_angle(i),
                step_i              => steps(i),
                x_coord_i           => x_coord(i),
                y_coord_i           => y_coord(i),
                rotation_angle_o    => rotation_angle(i+1),
                x_coord_o           => x_coord(i+1),
                y_coord_o           => y_coord(i+1)
            );
    end generate rotate_by_cordic_step_g;
    comb_g: if g_latency=0 generate
        x_coord_reg         <= x_coord       (c_nr_of_iterations_per_period);
        y_coord_reg         <= y_coord       (c_nr_of_iterations_per_period);
        rotation_angle_reg  <= rotation_angle(c_nr_of_iterations_per_period);
        ready_cordic_o      <= start_cordic_i;
    end generate comb_g;
    clocked_g: if g_latency/=0 generate
        process (res_i, clk_i)
        begin
            if res_i='1' then
                x_coord_reg        <= (others => '0');
                y_coord_reg        <= (others => '0');
                rotation_angle_reg <= (others => '0');
            elsif rising_edge(clk_i) then
                if enable_reg='1' then
                    x_coord_reg        <= x_coord       (c_nr_of_iterations_per_period);
                    y_coord_reg        <= y_coord       (c_nr_of_iterations_per_period);
                    rotation_angle_reg <= rotation_angle(c_nr_of_iterations_per_period);
                end if;
            end if;
        end process;
        ready_cordic_o <= ready_steps;
    end generate clocked_g;
    x_coord_o        <= x_coord_reg(x_coord_reg'high downto x_coord_reg'high-g_coordinate_width-1);
    y_coord_o        <= y_coord_reg(y_coord_reg'high downto y_coord_reg'high-g_coordinate_width-1);
    rotation_angle_o <= rotation_angle_reg(rotation_angle_reg'high downto
                                           rotation_angle_reg'high-g_angle_width+1);
end architecture;
