-- Filename: rotate_by_90_struct.vhd
-- Created by HDL-SCHEM-Editor at Sun Mar 10 11:18:04 2024
architecture struct of rotate_by_90 is
    constant c_plus_90_degree  : signed(g_angle_width-1 downto 0) := ('0', '1', others => '0');
    constant c_minus_90_degree : signed(g_angle_width-1 downto 0) := ('1', '1', others => '0');

    procedure rotate_vector (
        constant negative_direction_i : in  std_logic;
        constant x_i                  : in  signed(g_coordinate_width-1 downto 0);
        constant y_i                  : in  signed(g_coordinate_width-1 downto 0);
        signal   x_o                  : out signed(g_coordinate_width   downto 0);
        signal   y_o                  : out signed(g_coordinate_width   downto 0)
        ) is
    begin
        -- An additional bit is necessary for the case, when a coordinate is the smallest negative number and must be negated:
        if negative_direction_i='1' then
            x_o <=  (y_i(g_coordinate_width-1) & y_i);
            y_o <= -(x_i(g_coordinate_width-1) & x_i);
        else
            x_o <= -(y_i(g_coordinate_width-1) & y_i);
            y_o <=  (x_i(g_coordinate_width-1) & x_i);
        end if;
    end procedure;

    procedure set_rotation_angle(
        constant v_mode_i             : in  std_logic;
        constant negative_direction_i : in  std_logic;
        signal   rotation_angle_in    : in  signed(g_angle_width-1 downto 0);
        signal   rotation_angle_out   : out signed(g_angle_width-1 downto 0)
        ) is
    begin
        if v_mode_i='1' then -- rotation to x-axis
            if negative_direction_i='1' then
                rotation_angle_out <= c_plus_90_degree;
            else
                rotation_angle_out <= c_minus_90_degree;
            end if;
        else                 -- rotation by angle
            if negative_direction_i='1' then
                rotation_angle_out <= rotation_angle_in - c_minus_90_degree;
            else
                rotation_angle_out <= rotation_angle_in - c_plus_90_degree;
            end if;
        end if;
    end procedure;

    signal ready                        : std_logic;
    signal rotate_in_negative_direction : std_logic;
    signal rotation_angle               : signed(g_angle_width-1 downto 0);
    signal x_coord_rotated              : signed(g_coordinate_width downto 0);
    signal y_coord_rotated              : signed(g_coordinate_width downto 0);
begin
    -- Vector-Mode (rotation to x-axis):
    -- As the cordic algorithm only converges for rotations with less than 90 degree,
    -- vectors located in the second or third quadrant must be rotated into the first or fourth quadrant,
    -- before the rotation to the x-axis by the cordic algorithm can be started.
    -- Because the sign of the y-coordinate is used for identifying the quadrant the vector is located,
    -- also vectors from first and fourth quadrant are rotated, without any need, into fourth and first quadrant.
    -- But checking only the sign of the y-coordinate is the most simple condition available.
    -- Rotation Mode (rotation by rotation_angle_i):
    -- As the cordic algorithm only converges for rotations with less than 90 degree,
    -- rotation angles with an absolute value bigger than 90 degree must be reduced by 90 degree here,
    -- before the rotation by the cordic algorithm can be started.
    -- Because only the sign of the rotation angle is checked, also angles between -90...+90 are modified
    -- without any need. But checking only the sign of the rotation-angle is the most simple condition available.
    rotate_in_negative_direction <= '1' when (vector_mode_i='1' and y_coord_i(y_coord_i'high)              ='0') or
                                             (vector_mode_i='0' and rotation_angle_i(rotation_angle_i'high)='1') else
                                    '0';
    comb_g: if g_latency=0 generate 
        -- Rotation logic without flipflops:
        ready <= start_i;
        process(vector_mode_i, rotate_in_negative_direction, x_coord_i, y_coord_i, rotation_angle_i)
        begin
            rotate_vector      (rotate_in_negative_direction, x_coord_i, y_coord_i, x_coord_rotated, y_coord_rotated);
            set_rotation_angle (vector_mode_i, rotate_in_negative_direction, rotation_angle_i, rotation_angle);
        end process;
    end generate comb_g;
    clocked_g: if g_latency/=0 generate 
        -- Rotation logic:
        process(res_i, clk_i)
        begin
            if res_i='1' then
                x_coord_rotated <= (others => '0');
                y_coord_rotated <= (others => '0');
                rotation_angle  <= (others => '0');
                ready <= '0';
            elsif rising_edge(clk_i) then
                ready <= start_i;
                if start_i='1' then
                    rotate_vector      (rotate_in_negative_direction, x_coord_i, y_coord_i, x_coord_rotated, y_coord_rotated);
                    set_rotation_angle (vector_mode_i, rotate_in_negative_direction, rotation_angle_i, rotation_angle);
                end if;
            end if;
        end process;
    end generate clocked_g;
    -- Connect outputs:
    x_coord_rotated_o <= x_coord_rotated;
    y_coord_rotated_o <= y_coord_rotated;
    rotation_angle_o  <= rotation_angle;
    ready_o           <= ready;
end architecture;
