-- Filename: rotate_fix_cordic_length_struct.vhd
-- Created by HDL-SCHEM-Editor at Sun Mar 10 11:18:07 2024
architecture struct of rotate_fix_cordic_length is
    -- Calculation of the cos(delta) product:
    -- The cordic delta angles fulfill this equation: tan(delta) = 2**(-i), i = 0,1,2,3,...
    -- A right triangle with one cathetus of length 1 and the other cathetus of length 2**i will
    -- have a hypotenuse of length SQRT(1 + 2**2i) and will have an angle identical to delta.
    -- So the cosine of delta can be calculated by:
    -- cos(delta) = 2**i/SQRT(1 + 2**(2i))
    -- cos(delta) = 1/SQRT(2**(-2i) + 1)
    -- cos(delta) = 1/SQRT(1 + 2**(-2i))
    -- All the cos(delta) values from each micro-rotation must be multiplied.
    -- In this way the constant c_cordic_correction_factor was calculated by this Python3 program:
    --  import mpmath as mp_math
    --  mp_math.mp.dps = 200 # Number of digits of floating number
    --  # Check how many bits are available with the configured dps value:
    --  biggest_integer = 10**mp_math.mp.dps - 1
    --  number_of_bits_for_biggest_integer = mp_math.ceil(mp_math.log(biggest_integer, 2))
    --  number_of_bits_for_biggest_integer -= number_of_bits_for_biggest_integer%4 # Reduce the number of bits to make it dividable by 4.
    --  C_COORDINATE_WIDTH = number_of_bits_for_biggest_integer - 1
    --  product = 2 # First term with i=0
    --  for i in range(1,333): # 333 was selected, because the result did not change, when bigger numbers were used.
    --      product = mp_math.fmul(product, mp_math.fadd(1, mp_math.power(2,-2*i)))
    --  factor     = mp_math.sqrt(mp_math.fdiv(1, product))
    --  factor_hex = mp_math.fmul(factor, 2**(C_COORDINATE_WIDTH+1))
    --  print("{:0X}".format(int(factor_hex) & int(2**(C_COORDINATE_WIDTH+1)-1)))
    constant c_cordic_correction_factor: signed(664 downto 0) := '0'&X"9B74EDA8435E5A67F5F9092BD7FD40E9C288C51A3BD9F449B48004D6AE3ABF5E78983A2F7A3EE238F4F92F7EF93F4A079E2B30CEE59BF83FDDD640C4D7D887125DCD1A08A7CE513671EBB7362151288C842CED";
    signal x_coord_fixed : signed(2*g_coordinate_width+3 downto 0);
    signal y_coord_fixed : signed(2*g_coordinate_width+3 downto 0);
begin
    comb_g: if g_latency=0 generate 
        ready_fix_o   <= start_fix_i;
        x_coord_fixed <= x_coord_i * c_cordic_correction_factor(c_cordic_correction_factor'high downto c_cordic_correction_factor'high-g_coordinate_width-1);
        y_coord_fixed <= y_coord_i * c_cordic_correction_factor(c_cordic_correction_factor'high downto c_cordic_correction_factor'high-g_coordinate_width-1);
    end generate comb_g;
    clocked_g: if g_latency/=0 generate 
        process (res_i, clk_i)
        begin
            if res_i='1' then
                ready_fix_o   <= '0';
                x_coord_fixed <= (others => '0');
                y_coord_fixed <= (others => '0');
            elsif rising_edge(clk_i) then
                ready_fix_o <= start_fix_i;
                if start_fix_i='1' then
                    x_coord_fixed <= x_coord_i * c_cordic_correction_factor(c_cordic_correction_factor'high downto c_cordic_correction_factor'high-g_coordinate_width-1);
                    y_coord_fixed <= y_coord_i * c_cordic_correction_factor(c_cordic_correction_factor'high downto c_cordic_correction_factor'high-g_coordinate_width-1);
                end if;
            end if;
        end process;
    end generate clocked_g;
    -- The factors x_coord_i and c_cordic_correction_factor_i are signed datatypes with
    -- the first bit used as sign bit (two's complement).
    -- Therefore the product has always 2 sign bits (because c_cordic_correction_factor_i is always positive).
    -- From c_cordic_correction_factor_i g_coordinate_width+2 are used.
    -- These bits represent g_coordinate_width+1 bits after the binary point.
    -- So the bits g_coordinate_width:0 of the product can be ignored.
    -- The result must have a width of g_coordinate_width+1 bits, as the algorithm
    -- sometimes extends the incoming coordinates by 1 bit.
    -- So the product has 3 identical sign bits, 2 of them are cut off here:
    x_coord_o <= x_coord_fixed(2*g_coordinate_width+1 downto g_coordinate_width+1);
    y_coord_o <= y_coord_fixed(2*g_coordinate_width+1 downto g_coordinate_width+1);
end architecture;
