-- Filename: multiply_wt_booth_convert_struct.vhd
-- Created by HDL-SCHEM-Editor at Mon Jul 14 18:44:32 2025
architecture struct of multiply_wt_booth_convert is
begin
    process(multiplier_i)
        -- The multiplier must be extended by 1 MSB if it has an odd number of bits (an even number is needed for the conversion into radix 4).
        -- The multiplier must get an additional LSB with value 0 as "previous" bit for the first conversion.
        variable multiplier_ext : signed(2*g_multiplier_radix_4_width-1 downto -1);
    begin
        multiplier_ext := resize(multiplier_i, 2*g_multiplier_radix_4_width) & '0';
        -- The sign of the multiplier is ignored in the following (this causes an error at negative multipliers which is fixed later on when all partial products are calculated).
        -- So only the "data" bits are converted.
        -- In order to handle only the "data" bits, the sign bit must be replaced by 0 at negative multipliers:
        multiplier_ext(multiplier_ext'high) := '0';
        -- At the conversion 2 things are done to the multiplier:
        -- In the first step consecutive 1's are replaced by the Booth encoding.
        -- For example:
        -- ".  0  1  1  1  1  1  1  0  .  ." can be replaced by
        -- ".  1  0  0  0  0  0 -1  0  .  ."
        -- In the second step the radix is changed by packing 2 radix 2 digits (each having one of these values: -1,0,+1) into 1 radix 4 digit.
        -- This leads to a radix-4 digit with only 5 resulting values: -2, -1, 0, +1, +2 (coded in 2's complement)
        -- A multiplication of the multiplicand by these 5 factors can easily be implemented by shift and add operations.
        -- Both actions together can be implemented by the following "case" construct:
        for i in 0 to g_multiplier_radix_4_width-1 loop -- Calculate each radix-4 digit
            case multiplier_ext(2*i+1 downto 2*i-1) is -- Pick 3 bits of the multiplier: 2 bits to convert into a radix-4 digit and the previous one for a correct conversion.
                when "000"  => -- Old 2 bit value: 0
                    multiplier_radix_4_o(i) <= "000"; -- 0                                                                                                                                           => New radix-4 digit value :  0
                when "010"  => -- Old 2 bit value: 1
                    multiplier_radix_4_o(i) <= "001"; -- isolated 1                                                                                                                                  => New radix-4 digit value :  1
                when "110"  => -- Old 2 bit value: 3
                    multiplier_radix_4_o(i) <= "111"; -- Start of consecutive 1's in the lower bit                                                 => higher bit is set to 0, lower bit is set to -1 => New radix-4 digit value : -1
                when "100"  => -- Old 2 bit value: 2
                    multiplier_radix_4_o(i) <= "110"; -- Start of consecutive 1's in the higher bit                                                => higher bit is set to -1                        => New radix-4 digit value : -2
                when "111"  => -- Old 2 bit value: 3
                    multiplier_radix_4_o(i) <= "000"; -- Consecutive 1's                                                                           => higher and lower bit are both set to 0         => New radix-4 digit value :  0
                when "001"  => -- Old 2 bit value: 0
                    multiplier_radix_4_o(i) <= "001"; -- End of consecutive 1's in the previous bit                                                => lower bit is set to 1                          => New radix-4 digit value :  1
                when "011"  => -- Old 2 bit value: 1
                    multiplier_radix_4_o(i) <= "010"; -- End of consecutive 1's in the lower bit                                                   => higher bit is set to 1, lower bit is set to 0  => New radix-4 digit value :  2
                when "101"  => -- Old 2 bit value: 2
                    multiplier_radix_4_o(i) <= "111"; -- Start of consecutive 1's in the higher bit and end of consecutive 1's in the previous bit => higher bit is set to -1, lower bit is set to 1 => New radix-4 digit value : -2+1 = -1
                when others =>
                    null;
            end case;
        end loop;
    end process;
end architecture;
