library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package multiply_sd_package is
    -- SD (signed digit) numbers have digits with the values -3 .. +3 : "101", "110", "111", "000", "001", "010", "011".
    type t_sd_number is array (natural range <>) of signed(2 downto 0);

    function "+" (l: t_sd_number; r: t_sd_number) return t_sd_number;

    function sd_to_signed(sd_number: t_sd_number) return signed;

end package;

package body multiply_sd_package is
    function "+" (l: t_sd_number; r: t_sd_number) return t_sd_number is
        variable l_plus_r: signed(3 downto 0);
        variable ti      : signed(1 downto 0);
        variable wi      : signed(2 downto 0);
        variable result : t_sd_number(l'high downto 0);
    begin
        ti := "00";
        for i in 0 to l'high loop -- starting with LSB      -- Truthtable for the addition: abs(wi) shall never be bigger than 2.
            -- 4 bit adder for sign extended l and r:       -- l_plus_r(3:0) ti(1:0) wi(2:0), with 4*ti + wi = l_plus_r
            l_plus_r := (l(i)(2)&l(i)) + (r(i)(2)&r(i));    --  +6=0110      +1=01   +2=010
            -- Calculate the intermediate sum:              --  +5=0101      +1=01   +1=001
            wi(1 downto 0) := l_plus_r(1 downto 0);         --  +4=0100      +1=01    0=000
            if l_plus_r="0011" then                         --  +3=0011      +1=01   -1=111
                wi(2) := '1';                               --  +2=0010       0=00   +2=010
            elsif l_plus_r(3 downto 1)="110" then           --  +1=0001       0=00   +1=001
                wi(2) := '0';                               --   0=0000       0=00    0=000
            else                                            --  -1=1111       0=00   -1=111
                wi(2) := l_plus_r(3);                       --  -2=1110       0=00   -2=110
            end if;                                         --  -3=1101      -1=11   +1=001
            -- Calculate the complete sum:                  --  -4=1100      -1=11    0=000
            result(i) := resize(wi,3) + ti;                 --  -5=1011      -1=11   -1=111
            -- Calculate the transfer value:                --  -6=1010      -1=11   -1=110
            if l_plus_r(3 downto 1)="111" then
                ti(1) := '0';
            else
                ti(1) := l_plus_r(3);
            end if;
            if l_plus_r(3 downto 1)="111" or (l_plus_r(3 downto 2)="00" and l_plus_r(1 downto 0)/="11") then
                ti(0) := '0';
            else
                ti(0) := '1';
            end if;
        end loop;
        return result;
    end function;

    function sd_to_signed(sd_number: t_sd_number) return signed is
        variable positive_part : signed(2*sd_number'length-1 downto 0);
        variable negative_part : signed(2*sd_number'length-1 downto 0);
    begin
        for i in sd_number'range loop
            if sd_number(i)(2)='0' then
                positive_part(2*i+1 downto 2*i) := sd_number(i)(1 downto 0);
                negative_part(2*i+1 downto 2*i) := "00";
            else
                positive_part(2*i+1 downto 2*i) := "00";
                negative_part(2*i+1 downto 2*i) := -sd_number(i)(1 downto 0);
            end if;
        end loop;
        return positive_part - negative_part;
    end function;
end package body;
