-- Filename: multiply_karatsuba_step_struct.vhd
-- Created by HDL-SCHEM-Editor at Mon Apr  8 08:29:15 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of multiply_karatsuba_step is
    signal factor_1_high        : unsigned(g_operand_width/2-1 downto 0);
    signal factor_1_low         : unsigned(g_operand_width/2-1 downto 0);
    signal factor_2_high        : unsigned(g_operand_width/2-1 downto 0);
    signal factor_2_low         : unsigned(g_operand_width/2-1 downto 0);
    signal middle_term          : unsigned(g_operand_width downto 0);
    signal product_high         : unsigned(g_operand_width-1 downto 0);
    signal product_low          : unsigned(g_operand_width-1 downto 0);
    signal product_middle       : unsigned(g_operand_width downto 0);
    signal product_middle_pre   : unsigned(g_operand_width-1 downto 0);
    signal product_overflow_1   : unsigned(g_operand_width-1 downto 0);
    signal product_overflow_2   : unsigned(g_operand_width-1 downto 0);
    signal product_overflow_msb : unsigned(g_operand_width downto 0);
    signal sum1                 : unsigned(g_operand_width/2 downto 0);
    signal sum2                 : unsigned(g_operand_width/2 downto 0);
    signal sum_high_plus_low    : unsigned(g_operand_width downto 0);
    component multiply_karatsuba_step is
        generic (
            constant g_operand_width : natural := 16 -- must be a power of 2
        );
        port (
            factor_1_i : in  unsigned(g_operand_width-1 downto 0);
            factor_2_i : in  unsigned(g_operand_width-1 downto 0);
            product_o  : out unsigned(2*g_operand_width-1 downto 0)
        );
    end component;
begin
    karatsuba_g: if g_operand_width/=2 generate 
        factor_1_high <= factor_1_i(g_operand_width-1 downto factor_1_i'length/2);
        factor_1_low  <= factor_1_i(factor_1_i'length/2-1 downto 0);
        factor_2_high <= factor_2_i(g_operand_width-1 downto factor_2_i'length/2);
        factor_2_low  <= factor_2_i(factor_2_i'length/2-1 downto 0);
        sum1 <= unsigned('0'&factor_1_high) +
                unsigned('0'&factor_1_low );
        sum2 <= unsigned('0'&factor_2_high) +
                unsigned('0'&factor_2_low);
        multiply_karatsuba_step_high_inst : multiply_karatsuba_step
            generic map (
                g_operand_width => g_operand_width/2
            )
            port map (
                factor_1_i => factor_1_high,
                factor_2_i => factor_2_high,
                product_o  => product_high
            );
        multiply_karatsuba_step_low_inst : multiply_karatsuba_step
            generic map (
                g_operand_width => g_operand_width/2
            )
            port map (
                factor_1_i => factor_1_low,
                factor_2_i => factor_2_low,
                product_o  => product_low
            );
        multiply_karatsuba_step_middle_inst : multiply_karatsuba_step
            generic map (
                g_operand_width => g_operand_width/2
            )
            port map (
                factor_1_i => sum1(g_operand_width/2-1 downto 0),
                factor_2_i => sum2(g_operand_width/2-1 downto 0),
                product_o  => product_middle_pre
            );
        -- The product which must be calculated is sum1*sum2. As both sums got an additional overflow bit,
        -- the product must be calculated in this way:
        -- sum1 * sum2 =
        -- = (sum1(sum1'high)*2**g_operand_with/2 + sum1(g_operand_width/2-1 downto 0))*
        --   (sum2(sum2'high)*2**g_operand_with/2 + sum2(g_operand_width/2-1 downto 0))
        -- = sum1(sum1'high)*sum2(sum2'high)                   *2**g_operand_width +
        --   sum1(sum1'high)*sum2(g_operand_width/2-1 downto 0)*2**g_operand_width/2 +
        --   sum2(sum2'high)*sum1(g_operand_width/2-1 downto 0)*2**g_operand_width/2 +
        --   sum1(g_operand_width/2-1 downto 0)*sum2(g_operand_width/2-1 downto 0)
        --
        product_overflow_msb <= unsigned'('1' & (g_operand_width-1 downto 0 => '0'))
                                when sum1(sum1'high)='1' and sum2(sum2'high)='1' else
                                (others => '0');
        product_overflow_1   <= sum1(g_operand_width/2-1 downto 0)&(g_operand_width/2-1 downto 0 => '0')
                                when sum2(sum2'high)='1' else
                                (g_operand_width-1 downto 0 => '0');
        product_overflow_2   <= sum2(g_operand_width/2-1 downto 0)&(g_operand_width/2-1 downto 0 => '0')
                                when sum1(sum1'high)='1' else
                                (g_operand_width-1 downto 0 => '0');
        product_middle <= product_overflow_msb +
                          product_overflow_1   +
                          product_overflow_2   +
                          product_middle_pre;
        sum_high_plus_low <= unsigned('0'& product_high) +
                             unsigned('0'& product_low);
        middle_term <= product_middle - sum_high_plus_low;
        product_o <= (product_high & (g_operand_width  -1 downto 0 => '0')) +
                     (middle_term  & (g_operand_width/2-1 downto 0 => '0')) +
                      product_low;
    end generate karatsuba_g;
    last_g: if g_operand_width=2 generate 
        product_o <= unsigned(('0' & factor_1_i & '0') and (3 downto 0 => factor_2_i(1))) +
                     unsigned(('0' & '0' & factor_1_i) and (3 downto 0 => factor_2_i(0)));
    end generate last_g;
end architecture;
