library ieee;
use ieee.math_real.all;
-- Filename: multiply_bsc_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Thu Apr 24 16:48:35 2025
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use work.bsc_package.all;
architecture struct of multiply_bsc is
    function calculate_multiplier_bits_per_period return natural is
    begin
        if g_latency_mul=0 then
            return g_multiplier_width;
        end if;
        return integer(ceil(real(g_multiplier_width)/real(g_latency_mul)));
    end function;
    constant c_multiplier_bits_per_period : natural := calculate_multiplier_bits_per_period; 

    function calculate_multiplier_width_internal return natural is
    begin
        if g_latency_mul=0 then
            return g_multiplier_width;
        end if;
        return c_multiplier_bits_per_period*g_latency_mul;
    end function;
    constant c_multiplier_width_internal : natural := calculate_multiplier_width_internal;

    function calculate_number_of_periods return natural is
    begin
        if g_latency_mul=0 then
            return 1;
        end if;
        return g_latency_mul;
    end function;
    constant c_number_of_periods : natural := calculate_number_of_periods;

    type t_partial_product is array (natural range <>) of t_bsc_number(g_multiplicand_width downto 0);
    type t_shift_registers is array (natural range <>) of t_bsc_number(c_number_of_periods-1 downto 0);
    signal last_multiplier_bit                : std_logic_vector(c_multiplier_bits_per_period-1 downto 0);
    signal last_step                          : std_logic;
    signal multiplicand_bsc                   : t_bsc_number(g_multiplicand_width downto 0);
    signal multiplicand_neg                   : signed(g_multiplicand_width downto 0);
    signal multiplicand_neg_bsc               : t_bsc_number(g_multiplicand_width downto 0);
    signal multiplier_bits_from_shiftregister : signed(c_multiplier_bits_per_period-1 downto 0);
    signal multiplier_bits_used               : signed(c_multiplier_bits_per_period-1 downto 0);
    signal multiplier_int                     : signed(c_multiplier_width_internal-1 downto 0);
    signal partial_product                    : t_partial_product(c_multiplier_bits_per_period downto 0);
    signal partial_product_stored             : t_bsc_number(g_multiplicand_width-1 downto 0);
    signal product                            : t_bsc_number(g_multiplicand_width+g_multiplier_width-1 downto 0);
    signal product_high_part                  : t_bsc_number(g_multiplicand_width+g_multiplier_width-c_multiplier_width_internal-1 downto 0);
    signal product_low_part                   : t_bsc_number(c_multiplier_width_internal-1 downto 0);
    signal ready                              : std_logic;
    signal ready_mul                          : std_logic;
    signal reg_enable                         : std_logic;
    signal shift_registers                    : t_shift_registers(c_multiplier_bits_per_period-1 downto 0);
    component multiply_bsc_control is
        generic (
            g_counter_max : natural := 8
        );
        port (
            clk_i        : in  std_logic;
            res_i        : in  std_logic;
            start_i      : in  std_logic;
            last_step_o  : out std_logic;
            ready_o      : out std_logic;
            reg_enable_o : out std_logic
        );
    end component;
    component multiply_bsc_step is
        generic (
            g_multiplicand_width : natural := 8
        );
        port (
            factor_bit_i           : in  std_logic;
            last_multiplier_bit_i  : in  std_logic;
            multiplicand_bsc_i     : in  t_bsc_number(g_multiplicand_width downto 0);
            multiplicand_neg_bsc_i : in  t_bsc_number(g_multiplicand_width downto 0);
            partial_product_i      : in  t_bsc_number(g_multiplicand_width-1 downto 0);
            partial_product_o      : out t_bsc_number(g_multiplicand_width downto 0)
        );
    end component;
    component multiply_bsc_negate is
        generic (
            constant g_latency_mul        : natural := 0;
            constant g_multiplicand_width : natural := 8 
        );
        port (
            clk_i            : in  std_logic;
            multiplicand_i   : in  signed(g_multiplicand_width-1 downto 0);
            res_i            : in  std_logic;
            multiplicand_neg : out signed(g_multiplicand_width downto 0)
        );
    end component;
    component multiply_bsc_convert is
        generic (
            constant g_multiplicand_width : natural := 16;
            constant g_multiplier_width   : natural := 16;
            constant g_latency_convert    : natural :=  0 
        );
        port (
            clk_i     : in  std_logic;
            product_i : in  t_bsc_number(g_multiplicand_width+g_multiplier_width-1 downto 0);
            res_i     : in  std_logic;
            start_i   : in  std_logic;
            product_o : out signed(g_multiplicand_width+g_multiplier_width-1 downto 0);
            ready_o   : out std_logic
        );
    end component;
begin
    multiply_bsc_control_inst : multiply_bsc_control
        generic map (
            g_counter_max => c_number_of_periods-1
        )
        port map (
            clk_i        => clk_i,
            res_i        => res_i,
            start_i      => start_i,
            last_step_o  => last_step,
            ready_o      => ready,
            reg_enable_o => reg_enable
        );
    multiply_bsc_negate_inst : multiply_bsc_negate
        generic map (
            g_latency_mul        => g_latency_mul,
            g_multiplicand_width => g_multiplicand_width 
        )
        port map (
            clk_i            => clk_i,
            multiplicand_i   => multiplicand_i,
            res_i            => res_i,
            multiplicand_neg => multiplicand_neg
        );
    -- Convert the multiplicand from 2's complement into "binary stored carry" format (BSC) and
    -- add an additional bit for the use in multiply_bsc_step:
    multiplicand_bsc     <= unsigned_to_bsc(unsigned(multiplicand_i(g_multiplicand_width-1) & multiplicand_i));
    multiplicand_neg_bsc <= unsigned_to_bsc(unsigned(multiplicand_neg));
    -- When g_multiplier_width is not an integer multiple of g_latency_mul,
    -- then the multiplier must be extended by additional bits until its
    -- length is an integer multiple of g_latency_mul:
    multiplier_int <= resize(multiplier_i, multiplier_int'length);
    -- During the first step the multiplier bits are taken from the input multiplier_i, afterwards
    -- they are taken from the outputs of the shift registers:
    process (start_i, multiplier_i, multiplier_bits_from_shiftregister)
    begin
        if start_i='1' or g_latency_mul=0 or g_latency_mul=1 then
            multiplier_bits_used <= multiplier_i(c_multiplier_bits_per_period-1 downto 0);
        else
            multiplier_bits_used <= multiplier_bits_from_shiftregister;
        end if;
    end process;
    -- Multiplexer:
    partial_product(0)(g_multiplicand_width downto 1) <= (others => "00") when start_i='1' or
                                                                               g_latency_mul=0 or
                                                                               g_latency_mul=1 else
                                                          partial_product_stored;
    -- Bring the partial_product(0) into the same format as the output partial_product_o of module
    -- multiply_bsc_step has. The signal partial_product_stored misses the least significant
    -- digit (because it is already stored in shift_register), so here a dummy digit is added:
    partial_product(0)(0) <= "00";
    -- During multiplication the multiplier is handled as if it would be always a positive number.
    -- If the multiplier is indeed a positive number, everything is correct, as at all additions the
    -- sign of the multiplicand is handled in a correct way. The signbit of the positive multiplier has
    -- the value 0, so nothing is added in the step when this signbit is "multiplied" with
    -- the multiplicand.
    -- If the multiplier is a negative number then handling it as a positive number will not give
    -- the correct result. Because a negative number N is represented in 2's complement as 2**n+N (n=
    -- number of bits the 2's complent is using) the product would be:
    -- product = multiplicand * (2**n+N) = multiplicand*2**n + multiplicand*N
    -- This means the calculated product is too big by multiplicand*2**n.
    -- But this can be fixed easily during multiplication:
    -- When the multiplication reaches the sign bit of the multiplier and it has the value 1, then
    -- multiplicand*2**n must be subtracted from the result.
    -- So the module multiply_step must know, if the bit of the multiplier which is handled now, is the
    -- sign bit of the multiplier.
    -- This information is created here:
    process (last_step)
    begin
        last_multiplier_bit <= (others => '0');
        last_multiplier_bit(c_multiplier_bits_per_period-1) <= last_step;
    end process;
    multiply_step_g: for i in 0 to c_multiplier_bits_per_period-1 generate
        multiply_bsc_step_inst : multiply_bsc_step
            generic map (
                g_multiplicand_width => g_multiplicand_width
            )
            port map (
                factor_bit_i           => multiplier_bits_used(i),
                last_multiplier_bit_i  => last_multiplier_bit(i),
                multiplicand_bsc_i     => multiplicand_bsc,
                multiplicand_neg_bsc_i => multiplicand_neg_bsc,
                partial_product_i      => partial_product(i)(g_multiplicand_width downto 1),
                partial_product_o      => partial_product(i+1)
            );
    end generate multiply_step_g;
    register_g: if g_latency_mul/=0 generate
        process(res_i, clk_i)
        begin
            if res_i='1' then
                partial_product_stored <= (others => (others => '0'));
                shift_registers <= (others => (others => "00"));
            elsif rising_edge(clk_i) then
                if reg_enable='1' then
                    partial_product_stored <= partial_product(c_multiplier_bits_per_period)(g_multiplicand_width downto 1);
                    for i in 0 to c_multiplier_bits_per_period-1 loop
                        shift_registers(i)(c_number_of_periods-1)          <= partial_product(i+1)(0);
                        shift_registers(i)(c_number_of_periods-2 downto 0) <= shift_registers(i)(c_number_of_periods-1 downto 1);
                    end loop;
                end if;
                -- When start_i=1 then also reg_enable=1
                if start_i='1' then -- In the first step fill in all the multiplier bits at the lower bits of shift_registers:
                    for m in c_multiplier_bits_per_period to c_multiplier_width_internal-1 loop
                        shift_registers(m mod c_multiplier_bits_per_period)(m/c_multiplier_bits_per_period-1)(0) <= multiplier_int(m);
                        shift_registers(m mod c_multiplier_bits_per_period)(m/c_multiplier_bits_per_period-1)(1) <= '0';
                    end loop;
                end if;
            end if;
        end process;
        ready_mul <= ready;
    end generate register_g;
    combinatoric_g: if g_latency_mul=0 generate
        process(partial_product)
        begin
            partial_product_stored <= partial_product(c_multiplier_bits_per_period)(g_multiplicand_width downto 1);
            for i in 0 to c_multiplier_bits_per_period-1 loop
                shift_registers(i)(0) <= partial_product(i+1)(0);
            end loop;
        end process;
        ready_mul <= start_i;
    end generate combinatoric_g;
    process (shift_registers)
    begin
        for i in 0 to c_multiplier_bits_per_period-1 loop
            multiplier_bits_from_shiftregister(i) <= shift_registers(i)(0)(0);
        end loop;
    end process;
    product_high_part <= partial_product_stored(g_multiplicand_width-1-(c_multiplier_width_internal-g_multiplier_width) downto 0);
    process (shift_registers)
    begin
        for period_number in 0 to c_number_of_periods-1 loop
            for shift_register_number in 0 to c_multiplier_bits_per_period-1 loop
                product_low_part(c_multiplier_bits_per_period*period_number + shift_register_number) <= shift_registers(shift_register_number)(period_number);
            end loop;
        end loop;
    end process;
    -- If g_latency_mul is bigger than g_multiplicand_width+g_multiplier_width, then
    -- the complete product is stored in product_low_part and no bits from
    -- product_high_part are used:
    process(product_low_part, product_high_part)
    begin
        for i in 0 to g_multiplicand_width+g_multiplier_width-1 loop
            if i<c_multiplier_width_internal then
                product(i) <= product_low_part(i);
            else
                product(i) <= product_high_part(i-c_multiplier_width_internal);
            end if;
        end loop;
    end process;
    multiply_bsc_convert_inst : multiply_bsc_convert
        generic map (
            g_multiplicand_width => g_multiplicand_width,
            g_multiplier_width   => g_multiplier_width,
            g_latency_convert    => g_latency_convert
        )
        port map (
            clk_i     => clk_i,
            product_i => product,
            res_i     => res_i,
            start_i   => ready_mul,
            product_o => product_o,
            ready_o   => ready_o
        );
    process
    begin
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal partial_product_stored generate-condition == " & boolean'image(( g_latency_mul/=0 ));
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal partial_product_stored element-length = " & integer'image(partial_product_stored'length);
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal partial_product_stored(partial_product_stored'left) uses " & integer'image(partial_product_stored(partial_product_stored'left)'length) & " flipflops.";
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal shift_registers generate-condition == " & boolean'image(( g_latency_mul/=0 ));
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal shift_registers element-length = " & integer'image(shift_registers'length);
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal shift_registers element-length = " & integer'image(shift_registers(shift_registers'left)'length);
        report "flipflop_statistics for instance " & multiply_bsc'path_name & " signal shift_registers(shift_registers'left)(shift_registers(shift_registers'left)'left) uses " & integer'image(shift_registers(shift_registers'left)(shift_registers(shift_registers'left)'left)'length) & " flipflops.";
        wait;
    end process;
end architecture;
