library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package multiply_bsc_package is
    -- BSC numbers have digits with the values "00", "01", "10":
    type t_bsc_number is array (natural range <>) of unsigned(1 downto 0);

    function "+" (L, R : t_bsc_number) return t_bsc_number;
    function unsigned_to_bsc(unsigned_value: unsigned) return t_bsc_number;
    function bsc_to_unsigned(bsc_number: t_bsc_number) return unsigned;

end package;

package body multiply_bsc_package is
    function "+" (l, r : t_bsc_number) return t_bsc_number is
        -- Values for the next digit:
        variable ei     : unsigned(0 downto 0) := "0";  -- Initial value for digit 0
        variable ti     : unsigned(1 downto 0) := "00"; -- Initial value for digit 0
        -- Values of the actual digit:
        variable sum    : unsigned(2 downto 0);
        variable ti_tmp : unsigned(1 downto 0);
        variable wi     : unsigned(1 downto 0);
        variable result : t_bsc_number(l'high downto 0);
    begin
        for i in 0 to l'high loop -- starting with LSB
            -- Calculate the sum of the 2 digits,
            -- the sum can leave the range of BSC digits,
            -- so add an additional bit:
            sum       := ('0'&l(i)) + ('0'&r(i)); -- sum is in "000"..."100".
            -- Calculate the carry value for the next digit (do not
            -- assign it to ti, as the original ti is still needed):
            ti_tmp    := "+"(sum, ei)(2 downto 1); -- ti_tmp is in "000"..."101". The bits (2:1) are transfered to the next digit.
            -- Send the range of ti_tmp to the next digit (ei=0: low => ti_tmp = "0..1"; ei=1: high => ti_tmp = "1..2"):
            ei(0)     := sum(2) or sum(1); -- ei=1 when sum>=2
            -- Bring sum back into the legal range by subtracting 2*ti_temp (0,2,4) from sum.
            -- As after building the 2's complement of 2*ti_temp only the 2 lowest bits are needed, the subtracting is reduced to:
            wi        := sum(1 downto 0) + (ti_tmp(0)&'0'); -- wi is in -1,0,1 this means "11", "00", "01".
            -- Add the incoming transfer value which is in "00".."10".
            result(i) := wi + ti;
            -- Send the value of ti_tmp to the next digit:
            ti        := ti_tmp;
        end loop;
        return result;
    end function "+";
    function unsigned_to_bsc(unsigned_value: unsigned) return t_bsc_number is
        variable bsc_value : t_bsc_number(unsigned_value'range);
    begin
        for i in unsigned_value'range loop
            bsc_value(i)(1) := '0';
            bsc_value(i)(0) := unsigned_value(i);
        end loop;
        return bsc_value;
    end function;
    function bsc_to_unsigned(bsc_number: t_bsc_number) return unsigned is
        variable lower_bits  : unsigned(bsc_number'range);
        variable higher_bits : unsigned(bsc_number'range);
        variable result      : unsigned(bsc_number'high+1 downto 0);
    begin
        for i in bsc_number'range loop
            lower_bits(i)  := bsc_number(i)(0);
            higher_bits(i) := bsc_number(i)(1);
        end loop;
        result := higher_bits&'0' + lower_bits;
        return result(bsc_number'range);
    end function;
end package body;
