-- Filename: division_srt_radix2_step_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Sep 20 16:32:26 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of division_srt_radix2_step is
    signal carry_vector                      : signed(g_op_width-2 downto 0);
    signal carry_vector_shifted              : signed(g_op_width downto 0);
    signal double_partial_remainder_to_check : signed(3 downto 0);
    signal xor_vector                        : signed(g_op_width-1 downto 0);
begin
    -- Quotient bit selection by SRT algorithm:
    -- A radix2 SRT algorithm calculates the remainders R by this formula:
    --      R(i+1) = 2*R(i) - q(i)*Divisor           (1)
    --      with 2*R(0)=Dividend and q(i) in {-1,0,+1}
    -- The quotient bit q(i) must be selected in a way that always this condition is fulfilled:
    --      abs(R(i+1)) <= abs(Divisor)              (2)
    -- If q=0 is selected then equation (1) changes to
    --      R(i+1) = 2*R(i)                          (3)
    -- and condition (2) is only fulfilled, if abs(R(i)) is smaller than or equal to abs(Divisor/2).
    -- Therefore the condition for selecting q=0 is:
    --      abs(  R(i)) <= abs(Divisor/2)
    --      abs(2*R(i)) <= abs(Divisor  )            (4)
    -- This condition for selecting q=0 can be transformed:
    --      -abs(Divisor) <= 2*R(i) <= +abs(Divisor) (5)
    -- If 2*R(i) is outside the range of (5) and bigger than abs(Divisor), then 2*R(i) is positive and
    -- q=+1 must be selected for a positive divisor or q=-1 must be selected for a negative divisor in
    -- order to make R(i+1) smaller than abs(Divisor) in both cases.
    -- If 2*R(i) is outside the range of (5) and smaller than -abs(Divisor), then 2*R(i) is negative and
    -- q=-1 must be selected for a positive divisor or q=+1 must be selected for a negative divisor in
    -- order to make R(i+1) bigger than -abs(Divisor) in both cases.
    -- If 2*R(i) is inside the range of (5), q=0 is selected.
    -- As long as R(i) is always known exactly, there is no reason to select q=0 and equation (2) is
    -- always be fulfilled by using q=+/-1 (see module division_signed from this website).
    --
    -- But the SRT algorithm was invented in order to handle values of R(i) which are only approximately known.
    -- In SRT division the operands (provided in 2's complement) are normalized, which means they always start with "01..." or "10...".
    -- When R(i+1) is calculated by R(i+1)=2*R(i) (in the case when q=0), then R(i+1) will have 1 bit more than R(i).
    -- In the next step, in order to calculate R(i+2), first 2*R(i+1) is calculated, which means this product
    -- will have 2 bits more than R(i) and the calculation of the R(i)'s must use 2 bits more than the divisor has.
    -- Therefore the divisor will have values in the form "0001xxx" or "1110xxx" at the addition in equation (1).
    -- In order to accelerate this addition, carry-save adders are used, which speeds up addition a lot.
    -- So when equation (5) is checked, 2*R(i) is only available in form of a sum-part and a carry-part.
    -- In order to get an approximative value of 2*R(i) in a fast way, only the upper 4 bits of the sum-part and the carry-part are added.
    -- When the approximative value of 2*R(i) is "1111" or "0000" (and for example 2*R(i) has 8 bits) the range of 2*R(i) can be estimated:
    --   If 2*R(i) starts with "1111", then: "11110000" <= 2*R(i) <= "00000000"       (upper limit calculated by "11110000"+"00010000", with "00010000" being the maximum error provided by the estimation)
    --   If 2*R(i) starts with "0000", then: "00000000" <= 2*R(i) <= "00010000"       (upper limit calculated by "00000000"+"00010000")
    -- Both cases together mean:             "11110000" <= 2*R(i) <= "00010000"
    -- But as stated above the divisor has values between:    "11100000" <= divisor <= "11101111" (divisor lower range)
    --                                                     or "00010000" <= divisor <= "00011111" (divisor upper range)
    -- The range of 2*R(i) and the ranges of the divisor can be ordered:
    --        divisor lower range:      range of 2*R(i):        divisor upper range:
    --      "11100000"..."11101111", "11110000"..."00010000", "00010000"..."00011111"
    -- So when 2*R(i) starts with "1111" or "0000" the equation (5) is fulfilled and q=0 can be selected without knowing the exact value of 2*R(i).
    
    -- Calculate the upper 4 bits of 2*R(i) from the carry-save signals by adding:
    carry_vector_shifted              <= carry_vector_i & '0'; -- Shift to correct position.
    double_partial_remainder_to_check <= xor_vector_i        (g_op_width downto g_op_width-3)
                                       + carry_vector_shifted(g_op_width downto g_op_width-3);
    -- Check the partial remainder 2*R(i):
    process (double_partial_remainder_to_check, divisor_norm_i, xor_vector_i, carry_vector_i)
    begin
        if double_partial_remainder_to_check="1111" -- Instead of this if-structure SRT division often uses truthtables to
        or double_partial_remainder_to_check="0000" -- determine quotient_bit_plus_one_o and quotient_bit_minus_one_o.
        then                                        -- Here a truthtable with 9 inputs and 2 outputs would be needed.
            quotient_bit_plus_one_o  <= '0';
            quotient_bit_minus_one_o <= '0';
            xor_vector   <= xor_vector_i(g_op_width-1 downto 0) xor carry_vector_i(g_op_width-2 downto 0)&'0';                        -- Add only carry.
            carry_vector <= xor_vector_i(g_op_width-2 downto 0) and carry_vector_i(g_op_width-3 downto 0)&'0';
        elsif double_partial_remainder_to_check(3)=divisor_norm_i(divisor_norm_i'high) then
            quotient_bit_plus_one_o  <= '1';
            quotient_bit_minus_one_o <= '0';
            xor_vector   <= xor_vector_i(g_op_width-1 downto 0) xor carry_vector_i(g_op_width-2 downto 0)&'1' xor not divisor_norm_i; -- Add carry and inverted divisor.
            carry_vector <= (xor_vector_i  (g_op_width-2 downto 0)     and carry_vector_i(g_op_width-3 downto 0)&'1') or              -- Set LSB of carry vector to 1 for
                            (xor_vector_i  (g_op_width-2 downto 0)     and not divisor_norm_i(g_op_width-2 downto 0)) or              -- 2's complement of divisor.
                            (carry_vector_i(g_op_width-3 downto 0)&'1' and not divisor_norm_i(g_op_width-2 downto 0));
        else
            quotient_bit_plus_one_o  <= '0';
            quotient_bit_minus_one_o <= '1';
            xor_vector   <= xor_vector_i(g_op_width-1 downto 0) xor carry_vector_i(g_op_width-2 downto 0)&'0' xor divisor_norm_i;     -- Add carry and divisor.
            carry_vector <= (xor_vector_i  (g_op_width-2 downto 0)     and carry_vector_i(g_op_width-3 downto 0)&'0') or
                            (xor_vector_i  (g_op_width-2 downto 0)     and divisor_norm_i(g_op_width-2 downto 0)    ) or
                            (carry_vector_i(g_op_width-3 downto 0)&'0' and divisor_norm_i(g_op_width-2 downto 0)    );
        end if;
    end process;
    -- Double the results for multiplexing between dividend and 2*R(i):
    xor_vector_o   <= xor_vector   & '0';
    carry_vector_o <= carry_vector & '0';
end architecture;
