-- Filename: division_srt_radix2_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Oct 18 15:32:19 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of division_srt_radix2 is
    constant c_quotient_width : natural := g_dividend_width + 1 + g_add_quotient_bits;
    function get_max (op1 : natural; op2 : natural) return natural is
    begin
        if op1>=op2 then
            return op1;
        end if;
        return op2;
    end function;
    constant c_op_width : natural := get_max(g_dividend_width, g_divisor_width);
    -- The range of the signal quotient_too_big_by_shifts shall be limited, so a subtype is defined here for this signal:
    -- The signal quotient_too_big_by_shifts is calculated in the module division_srt_radix2_calc_shifts by this equation:
    --   quotient_too_big_by_shifts =   g_dividend_width - 1
    --                                - number_of_shifts_dvr
    --                                + number_of_shifts_dvd
    -- First the minimum value of quotient_too_big_by_shifts shall be calculated:
    -- Obviously number_of_shifts_dvd must be 0 in this case.
    -- When number_of_shifts_dvr has its maximum, which is g_divisor_width-1, there are 3 cases:
    -- Case 1: g_dividend_width=g_divisor_width:
    --   quotient_too_big_by_shifts =    g_dividend_width - 1
    --                                - (g_divisor_width  - 1) = 0
    -- So the minimum value for quotient_too_big_by_shifts is 0 in this case.
    -- Case 2: g_dividend_width>g_divisor_width,
    -- Then quotient_too_big_by_shifts is always bigger than 0.
    -- Case 3: g_dividend_width<g_divisor_width,
    -- then it seems as if quotient_too_big_by_shifts could get smaller than 0.
    -- But in this case number_of_shifts_dvd will not stay at the value 0, as required above,
    -- because the dividend will get N additional sign bits until the dividend has the same number of
    -- bits as the divisor and during normalization this leads to number_of_shifts_dvd=N as minimum:
    --   quotient_too_big_by_shifts =    g_dividend_width - 1
    --                                - (g_divisor_width  - 1)
    --                                + N
    -- But g_dividend_width+N is equal to g_divisor_width, so again the minimum value for quotient_too_big_by_shifts is 0.
    -- Second the maximum value of quotient_too_big_by_shifts shall be calculated:
    -- The maximum value of quotient_too_big_by_shifts is reached when number_of_shifts_dvr=0:
    --   quotient_too_big_by_shifts = g_dividend_width - 1
    --                                + number_of_shifts_dvd.
    -- When g_dividend_width>=g_divisor_width then the maximum of number_of_shifts_dvd is g_dividend_width-1.
    -- When g_divisor_width>g_dividend_width then the dividend gets additional bits until it has the same size as the divisor and
    -- can then be shifted at maximum by number_of_shifts_dvd=g_divisor_width-1.
    -- Because in this second case g_divisor_width>g_dividend_width, the second maximum exceeds the first one and
    -- the type definition must be:subtype t_quotient_too_big_by_shifts is integer range 0 to g_dividend_width + g_divisor_width - 2;
    subtype t_quotient_too_big_by_shifts  is integer range 0 to g_dividend_width + g_divisor_width - 2;
    -- The range of the signal remainder_too_big_by_shifts shall be limited, so a subtype is defined here for this signal:
    -- The signal remainder_too_big_by_shifts is calculated in the module division_srt_radix2_calc_shifts by this equation:
    -- remainder_too_big_by_shifts <= number_of_shifts_dvd_i-1 when number_of_shifts_dvd_i>number_of_shifts_dvr_i else
    --                                number_of_shifts_dvr_i;
    -- The minimum value of remainder_too_big_by_shifts is 0.
    -- The maximum value of number_of_shifts_dvd_i is g_dividend_width-1.
    -- The maximum value of number_of_shifts_dvr_i is g_divisor_width-1.
    -- So remainder_too_big_by_shifts has this range:
    subtype t_remainder_too_big_by_shifts is integer range 0 to get_max(g_dividend_width-2, g_divisor_width-1);
    signal dividend_norm               : signed(c_op_width-1 downto 0);
    signal divisor_norm                : signed(c_op_width-1 downto 0);
    signal number_of_shifts_dvd        : natural range 0 to g_dividend_width-1;
    signal number_of_shifts_dvr        : natural range 0 to g_divisor_width-1;
    signal quotient_norm               : signed(c_quotient_width-1 downto 0);
    signal quotient_too_big_by_shifts  : t_quotient_too_big_by_shifts;
    signal ready_div                   : std_logic;
    signal ready_norm                  : std_logic;
    signal remainder_norm              : signed(c_op_width-1 downto 0);
    signal remainder_too_big_by_shifts : t_remainder_too_big_by_shifts;
    component division_srt_radix2_core is
        generic (
            constant g_dividend_width   : natural := 8;
            constant g_divisor_width    : natural := 8;
            constant g_op_width         : natural := 8;
            constant g_add_quotient_bits: natural := 0;
            constant g_latency_division : natural := 8;
            constant g_latency_convert  : natural := 1
        );
        port (
            clk_i                  : in  std_logic;
            dividend_norm_i        : in  signed(g_op_width-1 downto 0);
            divisor_norm_i         : in  signed(g_op_width-1 downto 0);
            number_of_shifts_dvd_i : in  natural range 0 to g_dividend_width-1;
            number_of_shifts_dvr_i : in  natural range 0 to g_divisor_width-1;
            res_i                  : in  std_logic;
            start_i                : in  std_logic;
            quotient_norm_o        : out signed(g_dividend_width+g_add_quotient_bits downto 0);
            ready_o                : out std_logic;
            remainder_norm_o       : out signed(g_op_width-1 downto 0)
        );
    end component;
    component division_srt_radix2_norm is
        generic (
            constant g_op_width_org : natural := 8;
            constant g_op_width     : natural := 9;
            constant g_latency      : natural := 0
        );
        port (
            clk_i              : in  std_logic;
            op_i               : in  signed(g_op_width_org-1 downto 0);
            res_i              : in  std_logic;
            start_i            : in  std_logic;
            number_of_shifts_o : out natural range 0 to g_op_width-1 ;
            op_norm_o          : out signed(g_op_width-1 downto 0);
            ready_norm_o       : out std_logic
        );
    end component;
    component division_srt_radix2_denorm is
        generic (
            constant g_op_width          : natural := 16;
            constant g_shift_upper_limit : integer := 8;
            constant g_latency           : natural := 0
        );
        port (
            clk_i                        : in  std_logic;
            quotient_norm_i              : in  signed(g_op_width-1 downto 0);
            quotient_too_big_by_shifts_i : in  integer range 0 to g_shift_upper_limit;
            res_i                        : in  std_logic;
            start_i                      : in  std_logic;
            quotient_o                   : out signed(g_op_width-1 downto 0);
            ready_o                      : out std_logic
        );
    end component;
    component division_srt_radix2_denorm_remainder is
        generic (
            constant g_op_width          : natural := 8;
            constant g_shift_upper_limit : integer := 8;
            constant g_divisor_width     : integer := 8;
            constant g_latency           : natural := 0
        );
        port (
            clk_i                         : in  std_logic;
            remainder_norm_fixed_i        : in  signed(g_op_width-1 downto 0);
            remainder_too_big_by_shifts_i : in  integer range 0 to g_shift_upper_limit;
            res_i                         : in  std_logic;
            start_i                       : in  std_logic;
            rem_o                         : out signed(g_divisor_width-1 downto 0)
        );
    end component;
    component division_srt_radix2_calc_shifts is
        generic (
            constant g_dividend_width        : natural := 8;
            constant g_divisor_width         : natural := 8;
            constant g_shift_upper_limit     : integer := 8;
            constant g_shift_lower_limit     : integer := 0;
            constant g_shift_upper_limit_rem : integer := 8;
            constant g_shift_lower_limit_rem : integer := 0;
            constant g_latency_division      : integer := 0 
        );
        port (
            clk_i                         : in  std_logic;
            number_of_shifts_dvd_i        : in  natural range 0 to g_dividend_width-1;
            number_of_shifts_dvr_i        : in  natural range 0 to g_divisor_width-1;
            res_i                         : in  std_logic;
            quotient_too_big_by_shifts_o  : out integer range g_shift_lower_limit to g_shift_upper_limit;
            remainder_too_big_by_shifts_o : out integer range g_shift_lower_limit_rem to g_shift_upper_limit_rem
        );
    end component;
begin
    -- The signed dividend and the signed divisor are normalized here for SRT division.
    -- During normalisation the operand with less bits gets additional most significant bits, so that both operands have the same length.
    -- The normalized values will have only 1 sign bit (format: two's complement),
    -- this means all positive numbers start with "01..." and all negative numbers start with "10...".
    -- A value at a normalized operand is interpreted here (only in the comments) as a value which has a binary point after the first data bit.
    -- With this rule all positive from 0 different operands are in the range [+1,+2).
    -- With this rule all negative from 0 different operands are in the range [-2,-1).
    division_srt_radix2_norm_dvd_inst : division_srt_radix2_norm
        generic map (
            g_op_width_org => g_dividend_width,
            g_op_width     => c_op_width,
            g_latency      => g_latency_norm
        )
        port map (
            clk_i              => clk_i,
            op_i               => dividend_i,
            res_i              => res_i,
            start_i            => start_i,
            number_of_shifts_o => number_of_shifts_dvd,
            op_norm_o          => dividend_norm,
            ready_norm_o       => ready_norm
        );
    division_srt_radix2_norm_dvr_inst : division_srt_radix2_norm
        generic map (
            g_op_width_org => g_divisor_width,
            g_op_width     => c_op_width,
            g_latency      => g_latency_norm
        )
        port map (
            clk_i              => clk_i,
            op_i               => divisor_i,
            res_i              => res_i,
            start_i            => start_i,
            number_of_shifts_o => number_of_shifts_dvr,
            op_norm_o          => divisor_norm,
            ready_norm_o       => open
        );
    -- Because of the normalization a quotient different from 0, which is created by dividend and divisor having both the same sign, is in the range (+0.5, +2.0).
    -- Because of the normalization a quotient different from 0, which is created by dividend and divisor having different signs, is in the range (-2.0, -0.5).
    -- Because of the normalization the bit with index=c_quotient_width-2 of a positive quotient_norm has always the weight 2**0.
    -- So in order to get the correct results, quotient_norm and remainder_norm must be shifted in a way where this weight
    -- and the shifts provided by the normalization are taken into account.
    division_srt_radix2_calc_shifts_inst : division_srt_radix2_calc_shifts
        generic map (
            g_dividend_width        => g_dividend_width,
            g_divisor_width         => g_divisor_width,
            g_shift_upper_limit     => t_quotient_too_big_by_shifts'high,
            g_shift_lower_limit     => t_quotient_too_big_by_shifts'low,
            g_shift_upper_limit_rem => t_remainder_too_big_by_shifts'high,
            g_shift_lower_limit_rem => t_remainder_too_big_by_shifts'low,
            g_latency_division      => g_latency_division
        )
        port map (
            clk_i                         => clk_i,
            number_of_shifts_dvd_i        => number_of_shifts_dvd,
            number_of_shifts_dvr_i        => number_of_shifts_dvr,
            res_i                         => res_i,
            quotient_too_big_by_shifts_o  => quotient_too_big_by_shifts,
            remainder_too_big_by_shifts_o => remainder_too_big_by_shifts
        );
    division_srt_radix2_core_inst : division_srt_radix2_core
        generic map (
            g_op_width          => c_op_width,
            g_dividend_width    => g_dividend_width,
            g_divisor_width     => g_divisor_width,
            g_add_quotient_bits => g_add_quotient_bits,
            g_latency_division  => g_latency_division,
            g_latency_convert   => g_latency_convert
        )
        port map (
            clk_i                  => clk_i,
            dividend_norm_i        => dividend_norm,
            divisor_norm_i         => divisor_norm,
            number_of_shifts_dvd_i => number_of_shifts_dvd,
            number_of_shifts_dvr_i => number_of_shifts_dvr,
            res_i                  => res_i,
            start_i                => ready_norm,
            quotient_norm_o        => quotient_norm,
            ready_o                => ready_div,
            remainder_norm_o       => remainder_norm
        );
    division_srt_radix2_denorm_inst : division_srt_radix2_denorm
        generic map (
            g_op_width          => g_dividend_width + g_add_quotient_bits + 1,
            g_shift_upper_limit => t_quotient_too_big_by_shifts'high,
            g_latency           => g_latency_denorm
        )
        port map (
            clk_i                        => clk_i,
            quotient_norm_i              => quotient_norm,
            quotient_too_big_by_shifts_i => quotient_too_big_by_shifts,
            res_i                        => res_i,
            start_i                      => ready_div,
            quotient_o                   => quotient_o,
            ready_o                      => ready_o
        );
    division_srt_radix2_denorm_remainder_inst : division_srt_radix2_denorm_remainder
        generic map (
            g_op_width          => c_op_width,
            g_divisor_width     => g_divisor_width,
            g_shift_upper_limit => t_remainder_too_big_by_shifts'high,
            g_latency           => g_latency_denorm
        )
        port map (
            clk_i                         => clk_i,
            remainder_norm_fixed_i        => remainder_norm,
            remainder_too_big_by_shifts_i => remainder_too_big_by_shifts,
            res_i                         => res_i,
            start_i                       => ready_div,
            rem_o                         => rem_o
        );
end architecture;
