-- Filename: division_srt_radix2_core_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Oct 18 15:32:19 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
architecture struct of division_srt_radix2_core is
    constant c_quotient_width : natural := g_dividend_width + 1 + g_add_quotient_bits;
    function calculate_number_of_steps return natural is
    begin
        if g_latency_division=0 then
            return c_quotient_width-1;
        else
            return integer(ceil(real(c_quotient_width-1)/real(g_latency_division)));
        end if;
    end function;
    constant c_nr_of_steps   : natural := calculate_number_of_steps;
    function calculate_number_of_periods return natural is
    begin
        if g_latency_division=0 then
            return 1;
        else
            return g_latency_division;
        end if;
    end function;
    constant c_nr_of_periods : natural := calculate_number_of_periods;
    type t_xor_array   is array (natural range <>) of signed(g_op_width   downto 0);
    type t_carry_array is array (natural range <>) of signed(g_op_width-1 downto 0);
    type t_sr          is array (natural range <>) of unsigned(c_nr_of_periods-1 downto 0);
    subtype t_take_remainder_in_cycle is integer range 0 to g_divisor_width;
    signal carry_vector            : t_carry_array(c_nr_of_steps downto 0);
    signal carry_vector_reg        : signed(g_op_width-1 downto 0);
    signal carry_vector_rem        : signed(g_op_width-2 downto 0);
    signal counter                 : natural range 0 to c_nr_of_periods-1;
    signal quotient_bit_minus_one  : std_logic_vector(c_nr_of_steps-1 downto 0); -- A 1 represents a quotient bit with value '-1'.
    signal quotient_bit_plus_one   : std_logic_vector(c_nr_of_steps-1 downto 0); -- A 1 represents a quotient bit with value '+1'
    signal quotient_minus          : unsigned(c_nr_of_periods*c_nr_of_steps-1 downto 0);
    signal quotient_plus           : unsigned(c_nr_of_periods*c_nr_of_steps-1 downto 0);
    signal ready                   : std_logic;
    signal ready_div               : std_logic;
    signal reg_enable              : std_logic;
    signal remainder_pre_norm      : signed(g_op_width-1 downto 0);
    signal sr_minus_one            : t_sr(c_nr_of_steps-1 downto 0);
    signal sr_plus_one             : t_sr(c_nr_of_steps-1 downto 0);
    signal take_remainder_in_cycle : t_take_remainder_in_cycle;
    signal xor_vector              : t_xor_array(c_nr_of_steps downto 0);
    signal xor_vector_reg          : signed(g_op_width downto 0);
    signal xor_vector_rem          : signed(g_op_width-1 downto 0);
    component division_srt_radix2_control is
        generic (
            g_counter_max : natural := 8
        );
        port (
            clk_i        : in  std_logic;
            res_i        : in  std_logic;
            start_i      : in  std_logic;
            counter_o    : out natural range 0 to g_counter_max;
            ready_o      : out std_logic;
            reg_enable_o : out std_logic
        );
    end component;
    component division_srt_radix2_step is
        generic (
            constant g_op_width   : natural := 8
        );
        port (
            carry_vector_i           : in  signed(g_op_width-1 downto 0);
            divisor_norm_i           : in  signed(g_op_width-1 downto 0);
            xor_vector_i             : in  signed(g_op_width downto 0);
            carry_vector_o           : out signed(g_op_width-1 downto 0);
            quotient_bit_minus_one_o : out std_logic;
            quotient_bit_plus_one_o  : out std_logic;
            xor_vector_o             : out signed(g_op_width downto 0)
        );
    end component;
    component division_srt_radix2_convert is
        generic (
            constant g_quotient_width: natural := 8;
            constant g_op_width      : natural := 8;
            constant g_latency       : natural := 1 
        );
        port (
            carry_vector_rem_i   : in  signed(g_op_width-2 downto 0);
            clk_i                : in  std_logic;
            quotient_minus_i     : in  unsigned(g_quotient_width-1 downto 0);
            quotient_plus_i      : in  unsigned(g_quotient_width-1 downto 0);
            res_i                : in  std_logic;
            start_i              : in  std_logic;
            xor_vector_rem_i     : in  signed(g_op_width-1 downto 0);
            quotient_norm_o      : out signed(g_quotient_width downto 0);
            ready_o              : out std_logic;
            remainder_pre_norm_o : out signed(g_op_width-1 downto 0)
        );
    end component;
    component division_srt_radix2_fix_remainder is
        generic (
            constant g_op_width : natural := 8;
            constant g_dividend_width : natural := 8;
            constant g_divisor_width : natural := 8 
        );
        port (
            divisor_norm_i         : in  signed(g_op_width-1 downto 0);
            number_of_shifts_dvd_i : in  natural range 0 to g_dividend_width-1;
            number_of_shifts_dvr_i : in  natural range 0 to g_divisor_width-1;
            remainder_pre_norm_i   : in  signed(g_op_width-1 downto 0);
            sign_dividend_i        : in  std_logic;
            remainder_norm_o       : out signed(g_op_width-1 downto 0)
        );
    end component;
begin
    division_srt_radix2_control_inst : division_srt_radix2_control
        generic map (
            g_counter_max => c_nr_of_periods-1
        )
        port map (
            clk_i        => clk_i,
            res_i        => res_i,
            start_i      => start_i,
            counter_o    => counter,
            ready_o      => ready,
            reg_enable_o => reg_enable
        );
    -- Multiplexer between 2*R(i) and dividend:
    p_mux: process(start_i, dividend_norm_i, xor_vector_reg, carry_vector_reg)
    begin
        if start_i='1' or g_latency_division=0 or g_latency_division=1 then
            -- The normalized dividend is in the range [+1,+2) or [-2,-1).
            -- But the value transported by xor_vector and carry_vector is
            -- in the range [-4,4] at SRT division. Therefore the dividend
            -- must be extended by an additional sign bit:
            xor_vector  (0) <= dividend_norm_i(dividend_norm_i'high)&dividend_norm_i;
            carry_vector(0) <= (others => '0');
        else
            xor_vector  (0) <= xor_vector_reg;
            carry_vector(0) <= carry_vector_reg;
        end if;
    end process;
    division_srt_step_g: for i in 0 to c_nr_of_steps-1 generate
        division_srt_radix2_step_inst : division_srt_radix2_step
            generic map (
                g_op_width => g_op_width
            )
            port map (
                carry_vector_i           => carry_vector(i),
                divisor_norm_i           => divisor_norm_i,
                xor_vector_i             => xor_vector(i),
                carry_vector_o           => carry_vector(i+1),
                quotient_bit_minus_one_o => quotient_bit_minus_one(i),
                quotient_bit_plus_one_o  => quotient_bit_plus_one(i),
                xor_vector_o             => xor_vector(i+1)
            );
    end generate division_srt_step_g;
    take_remainder_in_cycle <= 0 when number_of_shifts_dvd_i>number_of_shifts_dvr_i else
                               1 + number_of_shifts_dvr_i - number_of_shifts_dvd_i;
    clocked_g: if g_latency_division/=0 generate
        ready_div <= ready;
        process (res_i, clk_i)
        begin
            if res_i='1' then
                xor_vector_reg   <= (others => '0');
                carry_vector_reg <= (others => '0');
                sr_plus_one      <= (others => (others => '0'));
                sr_minus_one     <= (others => (others => '0'));
                xor_vector_rem   <= (others => '0');
                carry_vector_rem <= (others => '0');
            elsif rising_edge(clk_i) then
                if start_i='1' or reg_enable='1' then
                    xor_vector_reg   <= xor_vector  (c_nr_of_steps);
                    carry_vector_reg <= carry_vector(c_nr_of_steps);
                    for i in 0 to c_nr_of_steps-1 loop
                        sr_plus_one (i) <= sr_plus_one (i)(c_nr_of_periods-2 downto 0) & quotient_bit_plus_one (i);
                        sr_minus_one(i) <= sr_minus_one(i)(c_nr_of_periods-2 downto 0) & quotient_bit_minus_one(i);
                        if c_nr_of_steps*counter+i=take_remainder_in_cycle then
                            -- The sum of xor_vector and carry_vector is equal to the doubled remainder,
                            -- so only half of the value is stored:
                            xor_vector_rem   <= xor_vector  (i)(g_op_width   downto 1);
                            carry_vector_rem <= carry_vector(i)(g_op_width-1 downto 1);
                        end if;
                    end loop;
                end if;
            end if;
        end process;
    end generate clocked_g;
    combinatorial_g: if g_latency_division=0 generate
        ready_div        <= start_i;
        xor_vector_reg   <= xor_vector  (c_nr_of_steps);
        carry_vector_reg <= carry_vector(c_nr_of_steps);
        process (quotient_bit_plus_one, quotient_bit_minus_one)
        begin
            for i in 0 to c_nr_of_steps-1 loop
                sr_plus_one (i)(0) <= quotient_bit_plus_one (i);
                sr_minus_one(i)(0) <= quotient_bit_minus_one(i);
            end loop;
        end process;
        -- The sum of xor_vector and carry_vector is equal to the doubled remainder, so only half of the value is stored:
        xor_vector_rem   <= xor_vector  (take_remainder_in_cycle)(g_op_width   downto 1);
        carry_vector_rem <= carry_vector(take_remainder_in_cycle)(g_op_width-1 downto 1);
    end generate combinatorial_g;
    -- Bring the quotient bits in the right order:
    p_copy: process(sr_plus_one, sr_minus_one)
    begin
        for p in c_nr_of_periods-1 downto 0 loop
            for s in 0 to c_nr_of_steps-1 loop
                quotient_plus (c_nr_of_steps*(p+1)-1-s) <= sr_plus_one (s)(p);
                quotient_minus(c_nr_of_steps*(p+1)-1-s) <= sr_minus_one(s)(p);
            end loop;
        end loop;
    end process;
    division_srt_radix2_convert_inst : division_srt_radix2_convert
        generic map (
            g_quotient_width => g_dividend_width + g_add_quotient_bits,
            g_latency        => g_latency_convert,
            g_op_width       => g_op_width
        )
        port map (
            carry_vector_rem_i   => carry_vector_rem,
            clk_i                => clk_i,
            quotient_minus_i     => quotient_minus(quotient_plus'high downto quotient_plus'high-c_quotient_width+2),
            quotient_plus_i      => quotient_plus(quotient_plus'high downto quotient_plus'high-c_quotient_width+2),
            res_i                => res_i,
            start_i              => ready_div,
            xor_vector_rem_i     => xor_vector_rem,
            quotient_norm_o      => quotient_norm_o,
            ready_o              => ready_o,
            remainder_pre_norm_o => remainder_pre_norm
        );
    division_srt_radix2_fix_remainder_inst : division_srt_radix2_fix_remainder
        generic map (
            g_op_width       => g_op_width,
            g_dividend_width => g_dividend_width,
            g_divisor_width  => g_divisor_width
        )
        port map (
            divisor_norm_i         => divisor_norm_i,
            number_of_shifts_dvd_i => number_of_shifts_dvd_i,
            number_of_shifts_dvr_i => number_of_shifts_dvr_i,
            remainder_pre_norm_i   => remainder_pre_norm,
            sign_dividend_i        => dividend_norm_i(g_op_width-1),
            remainder_norm_o       => remainder_norm_o
        );
end architecture;
