-- Filename: division_signed_step_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Jun 21 19:01:17 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of division_signed_step is
    signal interim_result : signed(g_divisor_width downto 0);
begin
    -- The division algorithm used here for signed dividend and divisor differs only slightly
    -- from the non restoring division with positive dividend and positive divisor.
    -- That algorithm switches at negative intermediate results from subtract to add, in order
    -- to fix the error done at the previous step at the current step.
    -- This criterion cannot be used with signed numbers anymore as a negative sign is not an
    -- indicator for having subtracted too much.
    --
    -- Instead the algorithm for signed operands compares the sign of the intermediate result and
    -- the sign of the divisor to switch between adding and subtracting. At the first step
    -- the sign of the dividend and the divisor are compared. When the signs are identical, then
    -- the divisor is subtracted, when they are different, the divisor is added.
    -- Both actions have the goal to reduce the dividend to 0 to by adding or subtracting
    -- multiples of the divisor.
    -- When a subtraction takes place, then the new quotient-bit gets the value "+1".
    -- When an addition takes places, then the new quotient-bit gets the value "-1" (which is coded by "0").
    -- The quotient in binary representation can afterwards be calculated by multiplying each
    -- quotient-bit (+1 or -1) with its weight 2^n (n = bitposition) and adding all these numbers.
    -- As the lowest quotient bit with weight 2^0 can only have the value +1 or -1 (as all other bits),
    -- the resulting quotient is always an odd number, which is a hint that a correction must
    -- be done. The problem is that the algorithm at the last iteration (as at any other iterations)
    -- only can set the quotient bit to +1 or -1. But sometimes the value 0 would have been needed.
    -- So the remainder must be checked afterwards and the quotient must be corrected.
    -- This is done, together with the conversion into the binary system, in the module division_signed_convert.
    interim_result <= interim_result_i & dividend_bit_i; -- "Draw" the next bit of the dividend.
    p_calc: process (interim_result, divisor_i)
    begin
        if interim_result(interim_result'high)=divisor_i(divisor_i'high) then
            quotient_bit_o   <= '1'; -- represents Value '+1'
            interim_result_o <= interim_result(g_divisor_width-1 downto 0) - divisor_i;
        else
            quotient_bit_o   <= '0'; -- represents Value '-1'
            interim_result_o <= interim_result(g_divisor_width-1 downto 0) + divisor_i;
        end if;
    end process;
end architecture;
