-- Filename: division_signed_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Oct 18 15:20:57 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
architecture struct of division_signed is
    function calculate_number_of_steps return natural is
    begin
        if g_latency_division=0 then
            return g_dividend_width-1; -- The sign bit is not processed in a step.
        else
            return integer(ceil(real(g_dividend_width-1)/real(g_latency_division)));
        end if;
    end function;
    constant c_nr_of_steps : natural := calculate_number_of_steps;
    function calculate_extended_dividend_width return natural is
    begin
        if g_latency_division=0 then
            return g_dividend_width;
        else
            return c_nr_of_steps*g_latency_division + 1; -- The sign bit causes the increment by 1.
        end if;
    end function;
    constant c_dividend_width_ext : natural := calculate_extended_dividend_width;
    constant c_nr_of_periods      : natural := (c_dividend_width_ext-1)/c_nr_of_steps;
    type t_interim_result is array (natural range <>) of signed(g_divisor_width-1 downto 0);
    type t_sr             is array (natural range <>) of signed(c_nr_of_periods-1 downto 0);
    signal dividend           : signed(c_dividend_width_ext-1 downto 0);
    signal dividend_bits      : signed(0 to c_nr_of_steps-1);
    signal interim_result     : t_interim_result(c_nr_of_steps downto 0);
    signal interim_result_reg : signed(g_divisor_width-1 downto 0);
    signal quotient_bit       : signed(c_nr_of_steps-1 downto 0);
    signal quotient_coded     : std_logic_vector(c_dividend_width_ext-2 downto 0);
    signal ready              : std_logic;
    signal ready_div          : std_logic;
    signal reg_enable         : std_logic;
    signal remainder          : signed(g_divisor_width-1 downto 0);
    signal sr                 : t_sr(c_nr_of_steps-1 downto 0);
    component division_signed_control is
        generic (
            g_counter_max : natural := 8
        );
        port (
            clk_i        : in  std_logic;
            res_i        : in  std_logic;
            start_i      : in  std_logic;
            ready_o      : out std_logic;
            reg_enable_o : out std_logic
        );
    end component;
    component division_signed_step is
        generic (
            constant g_divisor_width : natural := 8
        );
        port (
            dividend_bit_i   : in  std_logic;
            divisor_i        : in  signed(g_divisor_width-1 downto 0);
            interim_result_i : in  signed(g_divisor_width-1 downto 0);
            interim_result_o : out signed(g_divisor_width-1 downto 0);
            quotient_bit_o   : out std_logic
        );
    end component;
    component division_signed_convert is
        generic (
            constant g_latency            : natural := 1;
            constant g_dividend_width     : natural := 8;
            constant g_dividend_width_ext : natural := 8;
            constant g_divisor_width      : natural := 4 
        );
        port (
            clk_i            : in  std_logic;
            divisor_i        : in  signed(g_divisor_width-1 downto 0);
            quotient_coded_i : in  std_logic_vector(g_dividend_width_ext-2 downto 0);
            remainder_i      : in  signed(g_divisor_width-1 downto 0);
            res_i            : in  std_logic;
            sign_dvd_i       : in  std_logic;
            start_i          : in  std_logic;
            quotient_o       : out signed(g_dividend_width downto 0);
            ready_o          : out std_logic;
            remainder_o      : out signed(g_divisor_width-1 downto 0)
        );
    end component;
begin
    division_signed_control_inst : division_signed_control
        generic map (
            g_counter_max => c_nr_of_periods-1
        )
        port map (
            clk_i        => clk_i,
            res_i        => res_i,
            start_i      => start_i,
            ready_o      => ready,
            reg_enable_o => reg_enable
        );
    dividend <= resize(dividend_i, c_dividend_width_ext);
    p_mux: process(start_i, dividend, sr, interim_result_reg)
    begin
        if start_i='1' or g_latency_division=0 or g_latency_division=1 then
            -- Here the bits are swapped, e.g., dividend_bits(0)=dividend(c_dividend_width_ext-2):
            dividend_bits     <= dividend(c_dividend_width_ext-2 downto c_dividend_width_ext-1-c_nr_of_steps);
            interim_result(0) <= (others => dividend(c_dividend_width_ext-1));
        else
            for i in 0 to c_nr_of_steps-1 loop
                dividend_bits(i) <= sr(i)(c_nr_of_periods-1);
            end loop;
            interim_result(0) <= interim_result_reg;
        end if;
    end process;
    division_signed_step_g: for i in 0 to c_nr_of_steps-1 generate
        division_signed_step_inst : division_signed_step
            generic map (
                g_divisor_width => g_divisor_width
            )
            port map (
                dividend_bit_i   => dividend_bits(i),
                divisor_i        => divisor_i,
                interim_result_i => interim_result(i),
                interim_result_o => interim_result(i+1),
                quotient_bit_o   => quotient_bit(i)
            );
    end generate division_signed_step_g;
    clocked_g: if g_latency_division/=0 generate
        ready_div <= ready;
        process (res_i, clk_i)
        begin
            if res_i='1' then
                sr <= (others => (others => '0'));
                interim_result_reg <= (others => '0');
            elsif rising_edge(clk_i) then
                if start_i='1' then
                    for i in 0 to c_nr_of_steps-1 loop
                        for b in c_nr_of_periods-1 downto 0 loop
                            if b/=0 then
                                sr(i)(b) <= dividend(b*c_nr_of_steps-1-i);
                            else
                                sr(i)(b) <= quotient_bit(i);
                            end if;
                        end loop;
                    end loop;
                elsif reg_enable='1' then
                    for i in 0 to c_nr_of_steps-1 loop
                        sr(i)    <= shift_left(sr(i),1);
                        sr(i)(0) <= quotient_bit(i);
                    end loop;
                end if;
                if start_i='1' or reg_enable='1' then
                    interim_result_reg <= interim_result(c_nr_of_steps);
                end if;
            end if;
        end process;
        remainder <= interim_result_reg;
    end generate clocked_g;
    combinatorial_g: if g_latency_division=0 generate
        ready_div <= start_i;
        process (quotient_bit)
        begin
            for i in 0 to c_nr_of_steps-1 loop
                sr(i)(0) <= quotient_bit(i);
            end loop;
        end process;
        remainder <= interim_result(c_nr_of_steps);
    end generate combinatorial_g;
    p_copy: process(sr)
    begin
        for s in 0 to c_nr_of_steps-1 loop
            for p in c_nr_of_periods-1 downto 0 loop
                -- quotient_coded has no sign bit, but only bits
                -- which represent "+1" or "-1":
                quotient_coded(c_nr_of_steps*(p+1)-1-s) <= sr(s)(p);
            end loop;
        end loop;
    end process;
    division_signed_convert_inst : division_signed_convert
        generic map (
            g_latency            => g_latency_convert,
            g_dividend_width     => g_dividend_width,
            g_dividend_width_ext => c_dividend_width_ext,
            g_divisor_width      => g_divisor_width
        )
        port map (
            clk_i            => clk_i,
            divisor_i        => divisor_i,
            quotient_coded_i => quotient_coded,
            remainder_i      => remainder,
            res_i            => res_i,
            sign_dvd_i       => dividend_i(g_dividend_width-1),
            start_i          => ready_div,
            quotient_o       => quotient_o,
            ready_o          => ready_o,
            remainder_o      => rem_o
        );
end architecture;
