-- Filename: division_signed_convert_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Sun Jun 23 20:06:40 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
architecture struct of division_signed_convert is
    -- Fixing of quotient and remainder with procedure fix_results():
    -- This division algorithm changes positive or negative dividends (or intermediate results) towards 0.
    -- If the remainder has the same sign as the dividend, then the algorithm has stopped in time
    -- and nothing must be fixed (except in 2 special situations, see case 3. and 4. below).
    --
    -- But when both signs differ, then fixes must be done.
    -- The fixing works as in the module division_signed_step:
    -- When the divisor must be subtracted, then the quotient bit is set to +1.
    -- When the divisor must be added, then the quotient bit is set to -1.
    --
    -- There are 2 cases, where both signs differ:
    -- 1. Dividend>0 and remainder<0:
    -- If the divisor is positive, then a negative remainder indicates that the divisor has been subtracted one time too many (noted as +1),
    -- and the divisor must be added to the remainder and the quotient must be decremented by 1 (Action 2 in VHDL).
    -- If the divisor is negative, then a negative remainder signals that the divisor has been added one time too many (noted as -1),
    -- and the divisor must be subtracted from the remainder and the quotient must be incremented by 1 (Action 1 VHDL).
    -- 2. Dividend<0 and remainder>0:
    -- If the divisor is positive, then a positive remainder indicates that the divisor has been added once too often (noted as -1),
    -- and the divisor must be subtracted from the remainder and the quotient must be incremented by 1 (Action 1 VHDL).
    -- If the divisor is negative, then a positive remainder indicates that the divisor has been subtracted once too often (noted as +1),
    -- and the divisor must be added to the remainder and the quotient must be decremented by 1 (Action 2 VHDL).
    --
    -- When dividend and remainder are both negative, either fix 1 nor fix 2 is active.
    -- But if additionally the remainder is equal to the divisor or to the negated divisor, a fix must be used:
    -- 3. dividend<0 and remainder<0 and remainder=divisor (divisor<0):
    -- The divisor is negative, this means, the divisor was subtracted one time too seldom,
    -- and so the divisor must be subtracted from the remainder (sets remainder to 0) and the quotient must be incremented by 1 (Action 1 in VHDL).
    -- 4. dividend<0 and remainder<0 and remainder=-divisor (divisor>0):
    -- The divisor is positive, this means, the divisor was added one time too seldom,
    -- and so the divisor must be added to the remainder (sets remainder to 0) and the quotient must be decremented by 1 (Action 2 in VHDL).
    -- 
    -- When dividend and remainder are both positive, none of the 4 fixes above is active.
    -- But in this case the remainder can never be equal to the the divisor or to the negated divisor:
    -- 5. dividend>0 and remainder>0 and remainder=divisor (divisor>0):
    -- The last remainder can be equal to the divisor, if the previous remainder had the value 0 (together with the "drawn" dividend bit) and
    -- the divisor was added. But the divisor is only added if the sign of the previous remainder and of the divisor differ.
    -- But both have the same sign.
    -- So the last remainder can only be equal to the divisor, if the previous remainder had the value 2*divisor (together with the "drawn"
    -- dividend bit) and so the divisor was subtracted.
    -- This means, the previous remainder before "drawing" the dividend bit was also equal to the divisor.
    -- This must be true even at the last previous remainder.
    -- But this value is at a positive dividend either ...01 or ...00 (several sign bits with value 0 and 1 data bit) and never 2 times the divisor.
    -- So this case 5. never happens.
    -- 6. dividend>0 and remainder>0 and remainder=-divisor (divisor<0):
    -- The last remainder can be equal to the negated divisor, if the previous remainder had the value 0 (together with the "drawn"
    -- dividend bit) and the divisor was subtracted.
    -- But the divisor is only subtracted if the sign of the previous remainder and of the divisor are equal.
    -- But they have a different sign.
    -- So the last remainder can only be equal to the negated divisor, if the previous remainder had the value -2*divisor (together with the "drawn"
    -- dividend bit) and so the divisor was added.
    -- This means, the previous remainder before "drawing" the dividend bit was also equal to the negated divisor.
    -- This must be true even at the last previous remainder.
    -- But this value is at a positive dividend either ...01 or ...00 (several sign bits with value 0 and 1 data bit) and never 2 times the negated divisor.
    -- So this case 6. never happens.
    procedure fix_results(
        signal divisor_in      : in  signed(g_divisor_width-1 downto 0);
        signal remainder_in    : in  signed(g_divisor_width-1 downto 0);
        signal sign_dvd_in     : in  std_logic;
        signal quotient_in     : in  signed(g_dividend_width downto 0);
        signal quotient_fixed  : out signed(g_dividend_width downto 0);
        signal remainder_fixed : out signed(g_divisor_width-1 downto 0)
        ) is
        variable sign_dvr : std_logic;
        variable sign_rem : std_logic;
    begin
        sign_dvr := divisor_in(divisor_in'high);
        sign_rem := remainder_in(remainder_in'high);
        if    (remainder_in=divisor_in and sign_rem='1')
        or    (remainder_in/=0 and sign_rem/=sign_dvd_in and sign_dvr/=sign_dvd_in) then
            -- Action 1:
            quotient_fixed  <= quotient_in(g_dividend_width downto 0) + 1;
            remainder_fixed <= remainder_in - divisor_in;
        elsif (remainder_in=-divisor_in and sign_rem='1')
        or    (remainder_in/=0 and sign_rem/=sign_dvd_in and sign_dvr=sign_dvd_in) then
            -- Action 2:
            quotient_fixed  <= quotient_in(g_dividend_width downto 0) - 1;
            remainder_fixed <= remainder_in + divisor_in;
        else
            quotient_fixed  <= quotient_in(g_dividend_width downto 0);
            remainder_fixed <= remainder_in;
        end if;
    end procedure;
    signal quotient : signed(g_dividend_width_ext downto 0);
begin
    -- Explanation of the conversion of quotient_coded_i into quotient:
    -- The input quotient_coded_i is coded in this way:
    -- A bit with value 1 at index k represents the value "+1 * 2^k".
    -- A bit with value 0 at index k represents the value "-1 * 2^k".
    -- Be aware that the most significant bit of quotient_coded_i is not a sign bit but also a data bit.
    --
    -- This coding must be converted in the usual binary two's complement coding.
    -- When a bit "B" with the value +1 or -1 shall be expressed as a bit "b" with the value 0 or 1, this equation can be used:
    -- B = b*2 -1 with b=0 (when B=-1) or b=1 (when B=+1)
    -- A value "ABC" at quotient_coded_i can now be transformed:
    -- ABC =       A*2^2       + B*2^1       + C*2^0
    --     = (a*2-1)*2^2 + (b*2-1)*2^1 + (c*2-1)*2^0
    --     = a*2^3 - 2^2 + b*2^2 - 2^1 + c*2^1 - 2^0
    --     = a*2^3 + b*2^2 + c*2^1 - 2^2 - 2^1 - 2^0
    --     = abc0 - 0111     Usual binary coding!
    --     = abc0 + 1001     Instead of subtracting 0111, the two's complement of 0111 is added
    --
    -- This means:
    -- When the value ABC shall be converted into the usual binary coding these steps have to be performed:
    -- When a bit at ABC has the value +1 it must be replaced by 1, this means the bit can be used as it is.
    -- When a bit at ABC has the value -1 it must be replaced by 0 and as the value -1 is coded with 0, again the bit can be used as it is.
    -- The result abc must be shifted by 1 bit position to the left and a 0 must be filled in from the right side.
    -- Instead of adding 1001, the most significant bit is inverted and the least signficant bit is switched from 0 to 1.
    -- Be aware that the resulting binary value will always be a odd number!
    -- This is a hint, that for some results there is an addtional correction necessary (see below)!
    --
    -- Convert +1/-1 coding into binary coding:
    quotient <= signed (
                not quotient_coded_i(g_dividend_width_ext-2)          & -- add an additional sign bit (needed at procedure "fix_results"
                not quotient_coded_i(g_dividend_width_ext-2)          & --                        when g_dividend_width_ext=g_dividend_width)
                    quotient_coded_i(g_dividend_width_ext-3 downto 0) &
                    '1');
    clocked_g: if g_latency/=0 generate
        process (res_i, clk_i)
        begin
            if res_i='1' then
                ready_o     <= '0';
                quotient_o  <= (others => '0');
                remainder_o <= (others => '0');
            elsif rising_edge(clk_i) then
                ready_o <= start_i;
                if start_i='1' then
                    fix_results(divisor_i, remainder_i, sign_dvd_i, quotient, quotient_o, remainder_o);
                end if;
            end if;
        end process;
        
    end generate clocked_g;
    combinatorial_g: if g_latency=0 generate
        ready_o <= start_i;
        fix_results(divisor_i, remainder_i, sign_dvd_i, quotient, quotient_o, remainder_o);
    end generate combinatorial_g;
end architecture;
