-- Filename: division_unsigned_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Fri Oct 18 14:57:37 2024
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
library work;
architecture struct of division_unsigned is
    function calculate_number_of_steps return natural is
    begin
        if g_latency=0 then
            return g_dividend_width;
        else
            return integer(ceil(real(g_dividend_width)/real(g_latency)));
        end if;
    end function;
    constant c_nr_of_steps : natural := calculate_number_of_steps;
    function calculate_extended_dividend_width return natural is
    begin
        if g_latency=0 then
            return g_dividend_width;
        else
            return c_nr_of_steps*g_latency;
        end if;
    end function;
    constant c_dividend_width_ext : natural := calculate_extended_dividend_width;
    constant c_nr_of_periods      : natural := c_dividend_width_ext/c_nr_of_steps;
    type t_interim_result is array (natural range <>) of unsigned(g_divisor_width-1 downto 0);
    type t_sr             is array (natural range <>) of unsigned(c_nr_of_periods-1 downto 0);
    signal dividend           : unsigned(c_dividend_width_ext-1 downto 0);
    signal dividend_bits      : unsigned(0 to c_nr_of_steps-1);
    signal interim_result     : t_interim_result(c_nr_of_steps downto 0);
    signal interim_result_reg : unsigned(g_divisor_width-1 downto 0);
    signal quotient_bit       : unsigned(c_nr_of_steps downto 0);
    signal ready              : std_logic;
    signal reg_enable         : std_logic;
    signal sr                 : t_sr(c_nr_of_steps-1 downto 0);
    component division_unsigned_step is
        generic (
            constant g_divisor_width : natural := 8
        );
        port (
            dividend_bit_i      : in  std_logic;
            divisor_i           : in  unsigned(g_divisor_width-1 downto 0);
            interim_result_i    : in  unsigned(g_divisor_width-1 downto 0);
            quotient_bit_last_i : in  std_logic;
            interim_result_o    : out unsigned(g_divisor_width-1 downto 0);
            quotient_bit_o      : out std_logic
        );
    end component;
    component division_unsigned_control is
        generic (
            g_counter_max : natural := 8
        );
        port (
            clk_i        : in  std_logic;
            res_i        : in  std_logic;
            start_i      : in  std_logic;
            ready_o      : out std_logic;
            reg_enable_o : out std_logic
        );
    end component;
begin
    division_unsigned_control_inst : division_unsigned_control
        generic map (
            g_counter_max => c_nr_of_periods-1
        )
        port map (
            clk_i        => clk_i,
            res_i        => res_i,
            start_i      => start_i,
            ready_o      => ready,
            reg_enable_o => reg_enable
        );
    dividend <= resize(dividend_i, c_dividend_width_ext);
    p_mux: process(start_i, dividend, sr, interim_result_reg)
    begin
        if start_i='1' or g_latency=0 or g_latency=1 then
            -- Here the bits are swapped, e.g., dividend_bits(0)=dividend(c_dividend_width_ext-1):
            dividend_bits <= dividend(c_dividend_width_ext-1 downto c_dividend_width_ext-c_nr_of_steps);
            interim_result(0) <= (others => '0');
        else
            for i in 0 to c_nr_of_steps-1 loop
                dividend_bits(i) <= sr(i)(c_nr_of_periods-1);
            end loop;
            interim_result(0) <= interim_result_reg;
        end if;
    end process;
    quotient_bit(0) <= '1' when start_i='1' or g_latency=0 or g_latency=1 else
                       sr(c_nr_of_steps-1)(0);
    division_unsigned_step_g: for i in 0 to c_nr_of_steps-1 generate
        for division_unsigned_step_inst : division_unsigned_step use entity work.division_unsigned_step(struct);
    begin
        division_unsigned_step_inst : division_unsigned_step
            generic map (
                g_divisor_width => g_divisor_width
            )
            port map (
                dividend_bit_i      => dividend_bits(i),
                divisor_i           => divisor_i,
                interim_result_i    => interim_result(i),
                quotient_bit_last_i => quotient_bit(i),
                interim_result_o    => interim_result(i+1),
                quotient_bit_o      => quotient_bit(i+1)
            );
    end generate division_unsigned_step_g;
    clocked_g: if g_latency/=0 generate
        ready_o <= ready;
        process (res_i, clk_i)
        begin
            if res_i='1' then
                sr <= (others => (others => '0'));
                interim_result_reg <= (others => '0');
            elsif rising_edge(clk_i) then
                if start_i='1' then
                    for i in 0 to c_nr_of_steps-1 loop
                        for b in c_nr_of_periods-1 downto 0 loop
                            if b/=0 then
                                sr(i)(b) <= dividend(b*c_nr_of_steps-1-i);
                            else
                                sr(i)(b) <= quotient_bit(i+1);
                            end if;
                        end loop;
                    end loop;
                elsif reg_enable='1' then
                    for i in 0 to c_nr_of_steps-1 loop
                        sr(i)    <= shift_left(sr(i),1);
                        sr(i)(0) <= quotient_bit(i+1);
                    end loop;
                end if;
                if start_i='1' or reg_enable='1' then
                    interim_result_reg <= interim_result(c_nr_of_steps);
                end if;
            end if;
        end process;
        rem_o <= interim_result_reg when sr(c_nr_of_steps-1)(0)='1' else
                 interim_result_reg + divisor_i;
    end generate clocked_g;
    combinatorial_g: if g_latency=0 generate
        ready_o <= start_i;
        process (quotient_bit)
        begin
            for i in 0 to c_nr_of_steps-1 loop
                sr(i)(0) <= quotient_bit(i+1);
            end loop;
        end process;
        rem_o <= interim_result(c_nr_of_steps) when quotient_bit(c_nr_of_steps)='1' else
                 interim_result(c_nr_of_steps) + divisor_i;
    end generate combinatorial_g;
    p_copy: process(sr)
    begin
        for s in 0 to c_nr_of_steps-1 loop
            for p in c_nr_of_periods-1 downto 0 loop
                if c_nr_of_steps*(p+1)-1-s<g_dividend_width then
                    quotient_o(c_nr_of_steps*(p+1)-1-s) <= sr(s)(p);
                end if;
            end loop;
        end loop;
    end process;
end architecture;
