-- Filename: cordic_square_root_e.vhd
-- Created by HDL-SCHEM-Editor at Sun Jan 12 11:59:11 2025
-- This module calculates the square root of the input radicand.
-- It expects the radicand to be a positive integer.
-- In order to increase the accuracy the input radicand is first shifted
-- to the left by N_shift_left bits, to make it as big as possible.
-- Then the square root is calculated and must afterwards be shifted to the right, to compensate the left shifts:
--     radicand_shifted = radicand * 2**N_shift_left
--     SQRT(radicand_shifted) = SQRT(radicand * 2**N_shift_left) = SQRT(radicand) * SQRT(2**N_shift_left)
--     SQRT(radicand) = SQRT(radicand_shifted)/SQRT(2**N_shift_left)
--     SQRT(radicand) = SQRT(radicand_shifted)/2**(N_shift_left/2)
-- The division by 2**(N_shift_left/2) is done by a shift right.
-- The number of shifts to the right is N_shift_left/2, so N_shift_left must be a even number.
-- Because of this even number radicand_shifted has this range:
-- "01...." <= radicand_shifted <= "11...."
--
-- The square root is calculated by the Hyperbolic Cordic algorithm in vectoring mode.
-- This algorithm handles radicand_shifted as it would be in this range:
-- 0.25 <= radicand_shifted < 1.0
-- Compared to the shifted integer radicand, this means a fictive division by 2**(number_of_bits_of_the_radicand).
-- So after calculating the square root (range 0.5..1.0) the result must be multiplied by 2**(number_of_bits_of_the_radicand/2).
-- So it is important that the input radicand is extended to have always an even number of bits,
-- because then this multiplication can be performed by shifts.
--
-- When a square root is calculated by the hyperbolic cordic algorithm,
-- the 2 needed operands x[0] and y[0] are generated from the input radicand
-- by adding and subtracting a special constant c_mod_value.
-- This constant is selected in a way that the resulting x-coordinate x[n]
-- is equal to the square root without any length fix.
-- The hyperbolic cordic algorithm calculates x[n] to:
--     x[n] = K * square_root(x[0]**2 - y[0]**2), with K = 0.8281..
-- The constant c_mod_value for creating the 2 coordinates x[0] and y[0] from the radicand has this value:
--     c_mod_value = 1/(4*K**2) = 0.3645..
-- Then the 2 coordinates are:
--     x[0] = radicand + c_mod_value
--     y[0] = radicand - c_mod_value
-- The algorithm calculates:
--     x[n] = K * square_root((radicand + c_mod_value)**2 - (radicand - c_mod_value)**2)
--          = K * square_root( radicand**2 + 2*radicand*c_mod_value + c_mod_value**2
--                            -radicand**2 + 2*radicand*c_mod_value - c_mod_value**2
--          = K * square_root(4*radicand*c_mod_value)
--          = K * square_root(4*radicand/(4*K**2))
--          = K * square_root(radicand/K**2)
--          = square_root(radicand)
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
entity cordic_square_root is
    generic (
        constant g_radicand_width           : natural := 32;    -- Allowed values    : 1..1328
        constant g_improved_accuracy        : boolean := false; -- Allowed values    : true/false
        constant g_latency_prepare          : natural :=  0;    -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_latency_shift_radicand   : natural :=  0;    -- Allowed values    : 0..1 (values >1 are set to 1)
        constant g_latency_rotate_by_cordic : natural := 16;    -- Recommended values: 0..(g_radicand_width + g_radicand_width mod 2)/2, allowed all
        constant g_latency_shift_back       : natural :=  1     -- Allowed values    : 0..1 (values >1 are set to 1)
    );
    port (
        clk_i               : in  std_logic;
        radicand_i          : in  unsigned(g_radicand_width-1 downto 0);
        res_i               : in  std_logic;
        start_i             : in  std_logic;
        ready_o             : out std_logic;
        square_root_fract_o : out unsigned(g_radicand_width/2-1+g_radicand_width mod 2 downto 0);
        square_root_o       : out unsigned(g_radicand_width/2+g_radicand_width mod 2 downto 0)
    );
end entity cordic_square_root;
