-- Filename: cordic_square_root_struct.vhd.vhd
-- Created by HDL-SCHEM-Editor at Sun Jan 12 11:59:11 2025
library ieee;
use ieee.math_real.all;
architecture struct of cordic_square_root is
    constant c_radicand_width_even  : natural := g_radicand_width + g_radicand_width mod 2;

    function calculate_number_of_additional_bits return natural is
        variable number_of_iterations      : real;
        variable number_of_additional_bits : natural;
    begin
        if g_improved_accuracy=false then
            return 0;
        end if;
        number_of_iterations := real(c_radicand_width_even);
        number_of_additional_bits := natural(ceil(log2(number_of_iterations)));
        if number_of_additional_bits<2 then
            number_of_additional_bits := 2;
        end if;
        return number_of_additional_bits;
    end function;
    constant c_additional_bits    : natural := calculate_number_of_additional_bits;
    constant c_radicand_width_ext : natural := c_radicand_width_even + c_additional_bits;
    signal radicand_m         : signed(c_radicand_width_ext+1 downto 0);
    signal radicand_p         : signed(c_radicand_width_ext+1 downto 0);
    signal radicand_shifted   : unsigned(c_radicand_width_even-1 downto 0);
    signal radicand_zero      : std_logic;
    signal ready_cordic       : std_logic;
    signal ready_prepare      : std_logic;
    signal ready_shift        : std_logic;
    signal shift_nr           : natural range 0 to c_radicand_width_ext/2-1;
    signal square_root_cordic : unsigned(c_radicand_width_ext downto 0);
    component cordic_square_root_rotate is
        generic (
            constant g_radicand_width_even      : natural := 18;
            constant g_radicand_width_ext       : natural := 18;
            constant g_latency_rotate_by_cordic : natural :=  9
        );
        port (
            clk_i                : in  std_logic;
            radicand_m_i         : in  signed(g_radicand_width_ext+1 downto 0);
            radicand_p_i         : in  signed(g_radicand_width_ext+1 downto 0);
            res_i                : in  std_logic;
            start_cordic_i       : in  std_logic;
            ready_cordic_o       : out std_logic;
            square_root_cordic_o : out unsigned(g_radicand_width_ext downto 0)
        );
    end component;
    component cordic_square_root_shift_radicand is
        generic (
            constant g_radicand_width         : natural := 16;
            constant g_radicand_width_even    : natural := 16;
            constant g_latency_shift_radicand : natural :=  0 
        );
        port (
            clk_i              : in  std_logic;
            radicand_i         : in  unsigned(g_radicand_width-1 downto 0);
            res_i              : in  std_logic;
            start_i            : in  std_logic;
            radicand_shifted_o : out unsigned(g_radicand_width_even - 1 downto 0);
            radicand_zero_o    : out std_logic;
            ready_o            : out std_logic;
            shift_nr_o         : out natural range 0 to g_radicand_width_even/2-1
        );
    end component;
    component cordic_square_root_shift_back is
        generic (
            constant g_radicand_width_even : natural := 32;
            constant g_radicand_width_ext  : natural := 32;
            constant g_latency_shift_back  : natural :=  0 
        );
        port (
            clk_i                : in  std_logic;
            radicand_zero_i      : in  std_logic;
            res_i                : in  std_logic;
            shift_nr_i           : in  natural range 0 to g_radicand_width_ext/2-1;
            square_root_cordic_i : in  unsigned(g_radicand_width_ext downto 0);
            start_i              : in  std_logic;
            ready_o              : out std_logic;
            square_root_fract_o  : out unsigned(g_radicand_width_even/2-1 downto 0);
            square_root_o        : out unsigned(g_radicand_width_even/2 downto 0)
        );
    end component;
    component cordic_square_root_prepare_operands is
        generic (
            constant g_radicand_width_even  : natural := 16;
            constant g_radicand_width_ext   : natural := 16;
            constant g_latency_prepare      : natural :=  0
        );
        port (
            clk_i              : in  std_logic;
            radicand_shifted_i : in  unsigned(g_radicand_width_even-1 downto 0);
            res_i              : in  std_logic;
            start_i            : in  std_logic;
            radicand_m_o       : out signed(g_radicand_width_ext+1 downto 0);
            radicand_p_o       : out signed(g_radicand_width_ext+1 downto 0);
            ready_o            : out std_logic
        );
    end component;
begin
    -- Increase the accuracy by shifting the incoming radicand.
    -- The radicand is shifted to the left as much as possible:
    cordic_square_root_shift_radicand_inst : cordic_square_root_shift_radicand
        generic map (
            g_radicand_width         => g_radicand_width,
            g_radicand_width_even    => c_radicand_width_even,
            g_latency_shift_radicand => g_latency_shift_radicand
        )
        port map (
            clk_i              => clk_i,
            radicand_i         => radicand_i,
            res_i              => res_i,
            start_i            => start_i,
            radicand_shifted_o => radicand_shifted,
            radicand_zero_o    => radicand_zero,
            ready_o            => ready_shift,
            shift_nr_o         => shift_nr
        );
    -- Create 2 operands from the input radicand, so that the result of the
    -- hyperbolic cordic algorithm will be the square root of the radicand.
    -- A sign and an overflow bit is added:
    cordic_square_root_prepare_operands_inst : cordic_square_root_prepare_operands
        generic map (
            g_radicand_width_even  => c_radicand_width_even,
            g_radicand_width_ext   => c_radicand_width_ext,
            g_latency_prepare      => g_latency_prepare
        )
        port map (
            clk_i              => clk_i,
            radicand_shifted_i => radicand_shifted,
            res_i              => res_i,
            start_i            => ready_shift,
            radicand_m_o       => radicand_m,
            radicand_p_o       => radicand_p,
            ready_o            => ready_prepare
        );
    -- Calculate the unsigned square root of the 2 signed operands:
    cordic_square_root_rotate_inst : cordic_square_root_rotate
        generic map (
            g_radicand_width_even      => c_radicand_width_even,
            g_radicand_width_ext       => c_radicand_width_ext,
            g_latency_rotate_by_cordic => g_latency_rotate_by_cordic
        )
        port map (
            clk_i                => clk_i,
            radicand_m_i         => radicand_m,
            radicand_p_i         => radicand_p,
            res_i                => res_i,
            start_cordic_i       => ready_prepare,
            ready_cordic_o       => ready_cordic,
            square_root_cordic_o => square_root_cordic
        );
    -- Shrink the result of the cordic algorithm by using the number of shifts,
    -- which were introduced by the shifting of the radicand_i input:
    cordic_square_root_shift_back_inst : cordic_square_root_shift_back
        generic map (
            g_radicand_width_even => c_radicand_width_even,
            g_radicand_width_ext  => c_radicand_width_ext,
            g_latency_shift_back  => g_latency_shift_back
        )
        port map (
            clk_i                => clk_i,
            radicand_zero_i      => radicand_zero,
            res_i                => res_i,
            shift_nr_i           => shift_nr,
            square_root_cordic_i => square_root_cordic,
            start_i              => ready_cordic,
            ready_o              => ready_o,
            square_root_fract_o  => square_root_fract_o,
            square_root_o        => square_root_o
        );
end architecture;
