-- Filename: ahb_master_struct.vhd
-- Created by HDL-SCHEM-Editor at Thu Dec  4 14:36:38 2025
library ieee;
use ieee.numeric_std.all;
architecture struct of ahb_master is
    constant c_master_idle : t_ahb_control(haddr (g_addr_width  -1 downto 0),
                                           hburst(g_hburst_width-1 downto 0),
                                           hprot (g_hprot_width -1 downto 0)) := (
            haddr     => (others => '0'),
            hwrite    => '0',
            hsize     => "000",
            hburst    => (others => '0'),
            hprot     => (others => '0'),
            htrans    => "00",
            hmastlock => '0');

    -- Constants for AHB control-signals:
    constant idle   : std_logic_vector(1 downto 0) := "00";
    constant busy   : std_logic_vector(1 downto 0) := "01";
    constant nonseq : std_logic_vector(1 downto 0) := "10";
    constant seq    : std_logic_vector(1 downto 0) := "11";

    constant write  : std_logic := '1';
    constant read   : std_logic := '0';

    constant byte   : std_logic_vector(2 downto 0) := "000";

    constant single : std_logic_vector(g_hburst_width-1 downto 0) := (others => '0');

    constant lock   : std_logic := '1';
    constant nolock : std_logic := '0';

    signal d_master_haddr     : std_logic_vector(g_addr_width-1 downto 0);
    signal d_master_hburst    : std_logic_vector(g_hburst_width-1 downto 0);
    signal d_master_hmastlock : std_logic;
    signal d_master_hprot     : std_logic_vector(g_hprot_width-1 downto 0);
    signal d_master_hsize     : std_logic_vector(2 downto 0);
    signal d_master_htrans    : std_logic_vector(1 downto 0);
    signal d_master_hwrite    : std_logic;
    signal stimulation_runs   : std_logic := '1';
begin
    -- process to drive the AHB master interface:
    process
        procedure execute_ahb_access(         repeat    : in positive;
                                              htrans    : in std_logic_vector(1 downto 0);
                                              hwrite    : in std_logic;
                                              haddr     : in std_logic_vector;
                                              hburst    : in std_logic_vector;
                                              hsize     : in std_logic_vector;
                                              hmastlock : in std_logic) is
        begin
            for i in 0 to repeat-1 loop
                hcontrol_o.htrans    <= htrans;
                hcontrol_o.hwrite    <= hwrite;
                hcontrol_o.haddr     <= haddr;
                hcontrol_o.hsize     <= hsize;
                hcontrol_o.hburst    <= hburst;
                hcontrol_o.hmastlock <= hmastlock;
                wait for 1 ns; -- For a save check of hready_i
                while hready_i='0' loop
                    wait for g_period;
                end loop;
                hwdata_o                              <= (others => '0')                                  after g_period;
                hwdata_o(   g_addr_width-1 downto  0) <= haddr                                            after g_period;
                hwdata_o(               18 downto 16) <= hsize                                            after g_period;
                hwdata_o(19+g_hburst_width downto 20) <= hburst                                           after g_period;
                hwdata_o(               25 downto 24) <= htrans                                           after g_period;
                hwdata_o(               31 downto 28) <= std_logic_vector(to_unsigned(g_master_number,4)) after g_period;
                wait for g_period - 1 ns;
            end loop;
        end procedure;
    begin
        hcontrol_o <= c_master_idle;
        hwdata_o   <= (others => '0');
        wait for 1 ns;
        wait for 10*g_period;
        -- At AHB write accesses the hcontrol-information (haddr, hsize, hburst, htrans) must be copied into the write data.
        -- Then the slave can check, if the correct write-data has reached the slave.
        -- It is also checked, if haddr is inside the slave address range.
        -- It is not checked, how long the master had to wait for this access.
        --
        -- 1. All masters access the same slave at the same time (slave does not insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<16 * g_period loop
            wait for g_period;
        end loop;
    
        -- 2. All masters access the same slave at the same time (slave inserts 1 wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<30 * g_period loop
            wait for g_period;
        end loop;
    
        -- 3. All masters access the same slave at the same time (slave inserts 2 wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1400", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1402", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<50 * g_period loop
            wait for g_period;
        end loop;
    
        -- 4. One masters accesses both slaves (slaves don't insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3400", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1240", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<60 * g_period loop
            wait for g_period;
        end loop;
    
        -- 5. One masters accesses both slaves (slave 0 inserts 1 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3401", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<75 * g_period loop
            wait for g_period;
        end loop;
    
        -- 6. One masters accesses both slaves (both slaves insert 1 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3481", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<85 * g_period loop
            wait for g_period;
        end loop;
    
        -- 7. One masters accesses both slaves (both slaves insert 2 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1401", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"34A1", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1401", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<100 * g_period loop
            wait for g_period;
        end loop;
    
        -- 8. Two masters access different slaves at the same time
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1241", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3401", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1241", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"3402", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1242", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3402", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1242", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<120 * g_period loop
            wait for g_period;
        end loop;
    
        -- 9. One master starts delayed
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, idle  , read , X"0002", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1242", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<160 * g_period loop
            wait for g_period;
        end loop;
    
        -- 10. All masters access the same slave at the same time (slave does not insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, read , X"1230", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1231", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, read , X"1232", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<176 * g_period loop
            wait for g_period;
        end loop;
    
        -- 11. All masters access the same slave at the same time (slave inserts 1 wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, read , X"1200", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, read , X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<190 * g_period loop
            wait for g_period;
        end loop;
    
        -- 12. All masters access the same slave at the same time (slave inserts 2 wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, read , X"1400", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, read , X"1402", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<210 * g_period loop
            wait for g_period;
        end loop;
    
        -- 13. One masters accesses both slaves (slaves don't insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, read , X"1230", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3400", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1240", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<220 * g_period loop
            wait for g_period;
        end loop;
    
        -- 14. One masters accesses both slaves (slave 0 inserts 1 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3401", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<245 * g_period loop
            wait for g_period;
        end loop;
    
        -- 15. One masters accesses both slaves (both slaves insert 1 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3481", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<305 * g_period loop
            wait for g_period;
        end loop;
    
        -- 16. One masters accesses both slaves (both slaves insert 2 wait state)
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1401", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"34A1", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1401", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<360 * g_period loop
            wait for g_period;
        end loop;
    
        -- 17. Two masters access different slaves at the same time
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1241", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3401", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1241", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, read , X"3402", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1242", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3402", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1242", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<400 * g_period loop
            wait for g_period;
        end loop;
    
        -- 18. One master starts delayed
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, read , X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"3401", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, idle  , write, X"0002", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"1242", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<500 * g_period loop
            wait for g_period;
        end loop;
    
        -- 19. Both masters access an address gap
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"0001", single, byte, nolock);
            execute_ahb_access(1, nonseq, read , X"A001", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, read , X"0002", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"A002", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<510 * g_period loop
            wait for g_period;
        end loop;
    
        -- 20. All masters access the same slave at the same time with a burst (slave does not insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1240", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1250", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1241", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1251", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1262", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<525 * g_period loop
            wait for g_period;
        end loop;
    
        -- 21. All masters access the same slave at the same time with a burst (slave does insert wait states)
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1240", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1200", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1201", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1251", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<550 * g_period loop
            wait for g_period;
        end loop;
    
        -- 22. Two masters send a burst and the remaining master sends single accesses
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1240", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1201", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1251", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<575 * g_period loop
            wait for g_period;
        end loop;
    
        -- 23. Two masters send a burst to different slaves, but switch to the other slave at the same time
        if g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1241", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"3405", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"3406", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"3401", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"3402", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1262", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<620 * g_period loop
            wait for g_period;
        end loop;
    
        -- 24. One master accesses with lock
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1240", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, lock  );
            execute_ahb_access(1, nonseq, write, X"1251", single, byte, lock  );
            execute_ahb_access(1, nonseq, write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1242", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1252", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<640 * g_period loop
            wait for g_period;
        end loop;
    
        -- 25. The second access of master 0 has to wait because of 2 reasons:
        -- The dataphase of the first access is prolonged and afterwards the valid address phase of the second access does not get a grant
        -- (visible as prolonged data phase of the access to address 1240, which is an address without prolonged data phase):
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1240", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<680 * g_period loop
            wait for g_period;
        end loop;
    
        -- 26. Master 2 is running a burst over slave borders, but master 0 gets immediately the bus:
        if g_master_number=1 then
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1241", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"3452", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"3462", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<720 * g_period loop
            wait for g_period;
        end loop;
    
        -- 27. Error response at an access which had to wait:
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"123F", single, byte, nolock); -- 2. access: answered with error response
            execute_ahb_access(1, nonseq, write, X"1240", single, byte, nolock); -- 4. access
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock); -- 3. access
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, lock  );
            execute_ahb_access(1, nonseq, write, X"1251", single, byte, lock  );
            execute_ahb_access(1, nonseq, write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock); -- 1. access
            execute_ahb_access(1, idle  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<740 * g_period loop
            wait for g_period;
        end loop;
    
        -- 28. The address phase for access to "A000" at master 1 must remain valid,
        --     even when a master (1 or 2) has to wait for its access to slave 0:
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"A000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1260", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock); --
            execute_ahb_access(1, nonseq, write, X"1201", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1251", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1261", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, nonseq, write, X"1232", single, byte, nolock);
            execute_ahb_access(1, nonseq  , write, X"1242", single, byte, nolock);
            execute_ahb_access(1, nonseq  , write, X"1252", single, byte, nolock);
            execute_ahb_access(1, idle    , write, X"1202", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<840 * g_period loop
            wait for g_period;
        end loop;
    
        -- 29. Slave 0 is during a prolonged data phase, when 3 slaves want to access:
        if g_master_number=0 then
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"12C0", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"12D0", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"12E0", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1211", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1221", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1231", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1242", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1252", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"1262", single, byte, nolock);
        elsif g_master_number=3 then
            execute_ahb_access(1, nonseq, write, X"1203", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
        end if;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<860 * g_period loop
            wait for g_period;
        end loop;
    
        -- 30. Random:
        for i in 0 to 9 loop
        if g_master_number=0 then
            execute_ahb_access(1, nonseq, write, X"1200", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1201", single, byte, nolock);
            execute_ahb_access(1, busy  , write, X"1202", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1203", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1404", single, byte, nolock);
            execute_ahb_access(1, idle  , write, X"0005", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1406", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1407", single, byte, nolock);
        elsif g_master_number=1 then
            execute_ahb_access(1, nonseq, write, X"1010", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3411", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"A212", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"A113", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1214", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"3515", single, byte, nolock);
        elsif g_master_number=2 then
            execute_ahb_access(1, idle  , write, X"0000", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1221", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1222", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"1223", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1224", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"A025", single, byte, nolock);
        elsif g_master_number=3 then
            execute_ahb_access(1, nonseq, write, X"1230", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"0031", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"0032", single, byte, nolock);
            execute_ahb_access(1, seq   , write, X"0033", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"1234", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"0035", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"0036", single, byte, nolock);
            execute_ahb_access(1, nonseq, write, X"A037", single, byte, nolock);
        end if;
        end loop;
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        -- wait until all masters are idle again:
        while now<860 * g_period loop
            wait for g_period;
        end loop;
    
    
    
        execute_ahb_access(1, idle, read, X"0000", single, "000", nolock);
        stimulation_runs <= '0';
        wait;
    end process;
    debug_record(hcontrol_o,
    d_master_haddr,
    d_master_hsize,
    d_master_hburst,
    d_master_hprot,
    d_master_htrans,
    d_master_hwrite,
    d_master_hmastlock
    );
    
    -- process to check the read data:
    process
        procedure check_hrdata(count_length_error_messsage : in natural;
                               htrans       : in std_logic_vector(1 downto 0);
                               hwrite       : in std_logic;
                               haddr        : in std_logic_vector;
                               hsize        : in std_logic_vector;
                               hburst       : in std_logic_vector;
                               hmastlock    : in std_logic) is
        variable slave_number : natural;
        begin
        slave_number := 12345678;
        for i in 0 to g_start_addresses'length-1 loop
            if haddr(g_addr_width-1 downto g_slave_address_width(i))=g_start_addresses(i)(g_addr_width-1 downto g_slave_address_width(i)) then
                slave_number := i;
            end if;
        end loop;
        report " Master " & integer'image(g_master_number) & ": Check access to slave " & integer'image(slave_number);
        if slave_number/=12345678 then
            assert slave_number=to_integer(unsigned(master_hrdata_i(31 downto 28))) report "Error in master "  & integer'image(g_master_number) &
                                                    " : the slave_number is wrong: the access to slave " & integer'image(slave_number) & " has a different slave number in the read data: " & integer'image(to_integer(unsigned(master_hrdata_i(31 downto 28)))) severity warning;
            assert haddr =master_hrdata_i(   g_addr_width-1 downto  0) report "Error in master "  & integer'image(g_master_number) &
                                                    " : haddr is wrong: the access to address "   & to_hstring(haddr)  & " has a different address in the read data: " & to_hstring(master_hrdata_i(g_addr_width-1 downto  0)) severity warning;
            assert hsize =master_hrdata_i(               18 downto 16) report "Error in master "  & integer'image(g_master_number) &
                                                    " : hsize is wrong: the access with hsize "   & to_hstring(hsize)  & " has a different hsize in the read data: "   & to_hstring(master_hrdata_i(            18 downto 16)) severity warning;
            assert hburst=master_hrdata_i(19+g_hburst_width downto 20) report "Error in master "  & integer'image(g_master_number) &
                                                    " : hburst is wrong: the access with hburst " & to_hstring(hburst) & " has a different hburst in the read data: "  & to_hstring(master_hrdata_i(            22 downto 20)) severity warning;
            assert htrans=master_hrdata_i(               25 downto 24) report "Error in master "  & integer'image(g_master_number) &
                                                    " : htrans is wrong: the access with htrans " & to_hstring(htrans) & " has a different htrans in the read data: "   & to_hstring(master_hrdata_i(            25 downto 24)) severity warning;
        else
            assert master_hresp_i='1' report "Error in master "  & integer'image(g_master_number) & " : The master accessed an address gap but no error response was created" severity warning;
            assert count_length_error_messsage=1 and master_hresp_i='1' report "Error in master "  & integer'image(g_master_number) &
                                                    " : The master accessed an address gap and the error response did not have the correct length of 2." severity warning;
        end if;
        end procedure;
        variable v_d_master_haddr     : std_logic_vector(g_addr_width-1 downto 0);
        variable v_d_master_hsize     : std_logic_vector(2 downto 0);
        variable v_d_master_hburst    : std_logic_vector(g_hburst_width-1 downto 0);
        variable v_d_master_hprot     : std_logic_vector(g_hprot_width-1 downto 0);
        variable v_d_master_htrans    : std_logic_vector(1 downto 0);
        variable v_d_master_hwrite    : std_logic;
        variable v_d_master_hmastlock : std_logic;
        variable count_length_error_messsage : natural;
    begin
        wait for 0.75 * g_period;
        while stimulation_runs='1' loop
            if hready_i='1' and d_master_htrans/=idle then --valid address phase
                v_d_master_haddr     := d_master_haddr;
                v_d_master_hsize     := d_master_hsize;
                v_d_master_hburst    := d_master_hburst;
                v_d_master_hprot     := d_master_hprot;
                v_d_master_htrans    := d_master_htrans;
                v_d_master_hwrite    := d_master_hwrite;
                v_d_master_hmastlock := d_master_hmastlock;
                count_length_error_messsage := 0;
                wait for 0.5*g_period; -- Then 1/4 of the first clock period of the data phase has passed.
                while hready_i='0' loop
                    if master_hresp_i='1' Then
                        count_length_error_messsage := count_length_error_messsage + 1;
                    end if;
                    wait for g_period;
                end loop;
                check_hrdata(count_length_error_messsage,
                             v_d_master_htrans, v_d_master_hwrite, v_d_master_haddr, v_d_master_hsize,
                             v_d_master_hburst, v_d_master_hmastlock);
                wait for 0.5*g_period; -- Then 3/4 of the clock period of the valid data phase has passed.
            else
                wait for g_period;
            end if;
        end loop;
        wait;
    end process;
    
    stimulation_runs_o <= stimulation_runs;
end architecture;
