-- Filename: testbench_ahb_multilayer_struct.vhd
-- Created by HDL-SCHEM-Editor at Wed Oct 15 18:45:13 2025
library ieee;
use ieee.std_logic_1164.all;
use work.ahb_multilayer_package.all;
architecture struct of testbench_ahb_multilayer is
    constant c_period : time := 10 ns;
    constant c_number_of_masters   : positive :=  4;
    constant c_number_of_slaves    : positive :=  3;
    constant c_addr_width          : positive := 16;
    constant c_hburst_width        : positive :=  3;
    constant c_hprot_width         : positive :=  4;
    constant c_data_width          : positive := 32;
    constant c_slave_address_width : t_positive_list(0 to c_number_of_slaves-1) := (12, 8, 12);
    constant c_start_addresses     : t_std_logic_vector_list(0 to c_number_of_slaves-1) := (X"1000", X"3400", X"A000");
    constant c_wait1_addresses     : t_std_logic_vector_list(0 to c_number_of_slaves-1) := (X"1200", X"3480", X"A100");
    constant c_wait2_addresses     : t_std_logic_vector_list(0 to c_number_of_slaves-1) := (X"1400", X"34A0", X"A200");

    signal d_master_haddr     : std_logic_vector(c_addr_width-1 downto 0);
    signal d_master_hsize     : std_logic_vector(2 downto 0);
    signal d_master_hburst    : std_logic_vector(2 downto 0);
    signal d_master_hprot     : std_logic_vector(3 downto 0);
    signal d_master_htrans    : std_logic_vector(1 downto 0);
    signal d_master_hwrite    : std_logic;
    signal d_master_hmastlock : std_logic;

    signal d_slave_haddr     : std_logic_vector(c_addr_width-1 downto 0);
    signal d_slave_hsize     : std_logic_vector(2 downto 0);
    signal d_slave_hburst    : std_logic_vector(2 downto 0);
    signal d_slave_hprot     : std_logic_vector(3 downto 0);
    signal d_slave_htrans    : std_logic_vector(1 downto 0);
    signal d_slave_hwrite    : std_logic;
    signal d_slave_hmastlock : std_logic;

    signal hclk             : std_logic := '0';
    signal hresetn          : std_logic;
    signal master_hcontrol  : t_ahb_control_array (c_number_of_masters-1 downto 0)(haddr(c_addr_width-1 downto 0), hburst(c_hburst_width-1 downto 0), hprot(c_hprot_width-1 downto 0));
    signal master_hrdata    : t_std_logic_vector_list (c_number_of_masters-1 downto 0)(c_data_width-1 downto 0);
    signal master_hready    : std_logic_vector (c_number_of_masters-1 downto 0);
    signal master_hresp     : std_logic_vector (c_number_of_masters-1 downto 0);
    signal master_hwdata    : t_std_logic_vector_list (c_number_of_masters-1 downto 0)(c_data_width-1 downto 0);
    signal run_simulation   : std_logic := '1';
    signal slave_hcontrol   : t_ahb_control_array(c_number_of_slaves-1 downto 0)(haddr(c_addr_width-1 downto 0), hburst(c_hburst_width-1 downto 0), hprot(c_hprot_width-1 downto 0));
    signal slave_hrdata     : t_std_logic_vector_list(c_number_of_slaves-1 downto 0)(c_data_width-1 downto 0);
    signal slave_hready     : std_logic_vector (c_number_of_slaves-1 downto 0);
    signal slave_hreadyout  : std_logic_vector (c_number_of_slaves-1 downto 0);
    signal slave_hresp      : std_logic_vector (c_number_of_slaves-1 downto 0);
    signal slave_hsel       : std_logic_vector (c_number_of_slaves-1 downto 0);
    signal slave_hwdata     : t_std_logic_vector_list (c_number_of_slaves-1 downto 0)(c_data_width-1 downto 0);
    signal stimulation_runs : std_logic_vector (c_number_of_masters-1 downto 0);
    component ahb_multilayer is
        generic (
            g_number_of_masters   : positive :=  3;
            g_number_of_slaves    : positive :=  2;
            g_addr_width          : positive := 32;
            g_hburst_width        : natural  :=  3;
            g_hprot_width         : natural  :=  4;
            g_data_width          : positive := 32;
            g_slave_address_width : t_positive_list(0 to g_number_of_slaves-1) := (12, 8);
            g_start_addresses     : t_std_logic_vector_list(0 to g_number_of_slaves-1)(g_addr_width-1 downto 0) := (x"00001000", x"00003400") 
        );
        port (
            hclk_i      : in  std_logic  ;
            hcontrol_i  : in  t_ahb_control_array (g_number_of_masters-1 downto 0) (haddr(g_addr_width-1 downto 0), hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
            hrdata_i    : in  t_std_logic_vector_list (g_number_of_slaves-1 downto 0)(g_data_width-1 downto 0) ;
            hreadyout_i : in  std_logic_vector (g_number_of_slaves-1 downto 0) ;
            hresetn_i   : in  std_logic  ;
            hresp_i     : in  std_logic_vector (g_number_of_slaves-1 downto 0) ;
            hwdata_i    : in  t_std_logic_vector_list (g_number_of_masters-1 downto 0)(g_data_width-1 downto 0) ;
            hcontrol_o  : out t_ahb_control_array (g_number_of_slaves-1 downto 0) (haddr(g_addr_width-1 downto 0), hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
            hrdata_o    : out t_std_logic_vector_list (g_number_of_masters-1 downto 0)(g_data_width-1 downto 0) ;
            hready_o    : out std_logic_vector (g_number_of_masters-1 downto 0) ;
            hreadyin_o  : out std_logic_vector (g_number_of_slaves-1 downto 0) ;
            hresp_o     : out std_logic_vector (g_number_of_masters-1 downto 0) ;
            hsel_o      : out std_logic_vector (g_number_of_slaves-1 downto 0) ;
            hwdata_o    : out t_std_logic_vector_list (g_number_of_slaves-1 downto 0)(g_data_width-1 downto 0) 
        );
    end component;
    component ahb_slave is
        generic (
            g_slave_number      : natural  :=  1;
            g_addr_width        : positive := 16; -- maximum is 16 (because the address is copied into hrdata_o(15:0)
            g_start_address     : std_logic_vector(g_addr_width-1 downto 0);
            g_wait1_address     : std_logic_vector(g_addr_width-1 downto 0);
            g_wait2_address     : std_logic_vector(g_addr_width-1 downto 0);
            g_slave_addr_width  : positive :=  8;
            g_hburst_width      : natural  :=  3;
            g_hprot_width       : natural  :=  4;
            g_data_width        : positive := 32  -- minimum is 32 (because the slave number is copied into hrdata_o(31:16)
        );
        port (
            hclk_i      : in  std_logic  ;
            hcontrol_i  : in  t_ahb_control  (haddr(g_addr_width-1 downto 0), hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
            hready_i    : in  std_logic  ;
            hresetn_i   : in  std_logic  ;
            hsel_i      : in  std_logic  ;
            hwdata_i    : in  std_logic_vector (g_data_width-1 downto 0) ;
            hrdata_o    : out std_logic_vector (g_data_width-1 downto 0) ;
            hreadyout_o : out std_logic  ;
            hresp_o     : out std_logic  
        );
    end component;
    component ahb_master is
        generic (
            g_period : time := 10 ns;
            g_master_number : natural := 0;
            g_addr_width    : natural := 16;
            g_data_width    : natural := 32;
            g_hburst_width  : positive :=  3;
            g_hprot_width   : positive :=  4;
            g_number_of_slaves    : positive := 2;
            g_slave_address_width : t_positive_list(0 to g_number_of_slaves-1) := (12, 8);
            g_start_addresses     : t_std_logic_vector_list(0 to g_number_of_slaves-1)(g_addr_width-1 downto 0) := (x"1000", x"3400") 
        );
        port (
            hready_i           : in  std_logic  ;
            master_hrdata_i    : in  std_logic_vector (g_data_width-1 downto 0) ;
            master_hresp_i     : in  std_logic  ;
            hcontrol_o         : out t_ahb_control  (haddr(g_addr_width-1 downto 0), hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
            hwdata_o           : out std_logic_vector (g_data_width-1 downto 0) ;
            stimulation_runs_o : out std_logic  
        );
    end component;
begin
    process
    begin
        for i in 0 to c_number_of_slaves-1 loop
            assert c_slave_address_width(i)<c_addr_width report
                   "Error: slave-address-width " & integer'image(c_slave_address_width(i)) &
                   " is not smaller than master-address-width " & integer'image(c_addr_width) severity error;
        end loop;
        --wait for 1000*c_period;
        while stimulation_runs/=(c_number_of_masters-1 downto 0 => '0') loop
            wait for c_period;
        end loop;
        run_simulation <= '0';
        wait;
    end process;
    hresetn <= '0', '1' after 1 ns;
    process
    begin
        while run_simulation='1' loop
            hclk <= not hclk;
            wait for c_period/2;
        end loop;
        wait;
    end process;
    ahb_master_g: for i in 0 to c_number_of_masters-1 generate
        ahb_master_inst : ahb_master
            generic map (
                g_period              => c_period,
                g_master_number       => i,
                g_addr_width          => c_addr_width,
                g_data_width          => c_data_width,
                g_hburst_width        => c_hburst_width,
                g_hprot_width         => c_hprot_width, 
                g_number_of_slaves    => c_number_of_slaves,
                g_slave_address_width => c_slave_address_width,
                g_start_addresses     => c_start_addresses
            )
            port map (
                hready_i           => master_hready(i),
                master_hrdata_i    => master_hrdata(i),
                master_hresp_i     => master_hresp(i),
                hcontrol_o         => master_hcontrol(i),
                hwdata_o           => master_hwdata(i),
                stimulation_runs_o => stimulation_runs(i)
            );
    end generate ahb_master_g;
    ahb_multilayer_inst : ahb_multilayer
        generic map (
            g_number_of_slaves    => c_number_of_slaves,
            g_addr_width          => c_addr_width,
            g_hburst_width        => c_hburst_width,
            g_hprot_width         => c_hprot_width,
            g_data_width          => c_data_width,
            g_slave_address_width => c_slave_address_width,
            g_number_of_masters   => c_number_of_masters,
            g_start_addresses     => c_start_addresses 
        )
        port map (
            hclk_i      => hclk,
            hcontrol_i  => master_hcontrol,
            hrdata_i    => slave_hrdata,
            hreadyout_i => slave_hreadyout,
            hresetn_i   => hresetn,
            hresp_i     => slave_hresp,
            hwdata_i    => master_hwdata,
            hcontrol_o  => slave_hcontrol,
            hrdata_o    => master_hrdata,
            hready_o    => master_hready,
            hreadyin_o  => slave_hready,
            hresp_o     => master_hresp,
            hsel_o      => slave_hsel,
            hwdata_o    => slave_hwdata
        );
    ahb_slave_g: for i in 0 to c_number_of_slaves -1 generate
        ahb_slave_inst : ahb_slave
            generic map (
                g_start_address     => c_start_addresses(i),
                g_slave_addr_width  => c_slave_address_width(i),
                g_slave_number      => i,
                g_addr_width        => c_addr_width, -- maximum is 16 (because the address is copied into hrdata_o(15:0)
                g_hburst_width      => c_hburst_width,
                g_hprot_width       => c_hprot_width,
                g_data_width        => c_data_width, -- minimum is 32 (because the slave number is copied into hrdata_o(31:16)
                g_wait1_address     => c_wait1_addresses(i),
                g_wait2_address     => c_wait2_addresses(i)
            )
            port map (
                hclk_i      => hclk,
                hcontrol_i  => slave_hcontrol(i),
                hready_i    => slave_hready(i),
                hresetn_i   => hresetn,
                hsel_i      => slave_hsel(i),
                hwdata_i    => slave_hwdata(i),
                hrdata_o    => slave_hrdata(i),
                hreadyout_o => slave_hreadyout(i),
                hresp_o     => slave_hresp(i)
            );
    end generate ahb_slave_g;
    debug_record(master_hcontrol(0),
    d_master_haddr,
    d_master_hsize,
    d_master_hburst,
    d_master_hprot,
    d_master_htrans,
    d_master_hwrite,
    d_master_hmastlock
    );
    debug_record(slave_hcontrol(0),
    d_slave_haddr,
    d_slave_hsize,
    d_slave_hburst,
    d_slave_hprot,
    d_slave_htrans,
    d_slave_hwrite,
    d_slave_hmastlock
    );
end architecture;
