-- Filename: ahb_multilayer_arbiter_struct.vhd
-- Created by HDL-SCHEM-Editor at Fri Nov 21 15:00:41 2025
architecture struct of ahb_multilayer_arbiter is
    signal filled_grant           : std_logic_vector(g_number_of_masters-1 downto 0);
    signal grant                  : std_logic_vector(g_number_of_masters-1 downto 0);
    signal grant_s                : std_logic_vector(g_number_of_masters-1 downto 0);
    signal mask                   : std_logic_vector(2*g_number_of_masters-1 downto 0);
    signal new_grant_doubled      : std_logic_vector(2*g_number_of_masters-1 downto 0);
    signal request_doubled        : std_logic_vector(2*g_number_of_masters-1 downto 0);
    signal request_doubled_masked : std_logic_vector(2*g_number_of_masters-1 downto 0);
begin
    -- In order to create the next grant first all bits left of the actual grant are filled with 1 (in signal filled_grant).
    -- The actual grant will only be visible in filled_grant (at the position with the highest priority),
    -- if the actual request is part of a burst or if the bus is locked:
    process(grant_s, use_last_grant_i)
        variable fill_with_1 : std_logic;
    begin
        fill_with_1 := '0';
        for i in 0 to g_number_of_masters-1 loop
            filled_grant(i) <= (grant_s(i) and use_last_grant_i(i)) or fill_with_1;
            fill_with_1 := fill_with_1 or grant_s(i);
        end loop;
    end process;
    -- Create the signal mask with 2*g_number_of_masters bits.
    -- Only g_number_of_master bits are filled with 1.
    -- The sequence of 1's starts at the actual grant or directly after the actual grant:
    mask <= not filled_grant & filled_grant;
    -- Double the requests to have the same number of bits as the mask:
    request_doubled <= request_i & request_i;
    -- The lowest 1 in request_doubled_masked is the new request which shall be granted:
    request_doubled_masked <= request_doubled and mask;
    -- Let only the lowest 1 in request_doubled_mask pass:
    process(request_doubled_masked)
        variable remove_request : std_logic;
    begin
        remove_request := '0';
        for i in 0 to 2*g_number_of_masters-1 loop
            new_grant_doubled(i) <= request_doubled_masked(i) and not remove_request;
            remove_request := remove_request or request_doubled_masked(i); -- The first found request sets remove_request.
        end loop;
    end process;
    -- Resize back to the original width:
    grant <= grant_s when request_i=(request_i'range => '0') else
             new_grant_doubled(2*g_number_of_masters-1 downto g_number_of_masters) or
             new_grant_doubled(  g_number_of_masters-1 downto 0);
    -- Connect the output:
    grant_o <= grant when g_number_of_masters/=1 else (others => '1'); -- Check for generic to make it simple for synthesis
    -- Store the new grant for the next arbitration:
    process(hresetn_i, hclk_i)
    begin
        if hresetn_i='0' then
            grant_s <= (others => '0');
            grant_s(0) <= '1';
        elsif rising_edge(hclk_i) then
            if hready_i='1' then -- a valid address phase is active
                grant_s <= grant;
            end if;
        end if;
    end process;
    grant_s_o <= grant_s when g_number_of_masters/=1 else (others => '1'); -- Check for generic to make it simple for synthesis
end architecture;
