-- Filename: ahb_multilayer_demux_hreadyout_struct.vhd
-- Created by HDL-SCHEM-Editor at Thu Oct 16 10:32:29 2025
--
-- Explanation for the generation of hreadyout_wait:
-- Assume a situation were 3 masters try to access the same slave at the same time, when
-- master 1 has the highest priority, master 3 the lowest.
--
-- Master 1 gets immediately a grant, which means its hcontrol signals are now connected
-- to the slave, which is in the last clock cycle of the data phase of the access before.
-- The hreadyout of the slave determines the hready at master 1 during the address phase.
--
-- Master 2 does not get a grant during this address phase and the signal hreadyout_wait
-- determines the value at hready at master 2 and has always the value 1 in this situation.

-- Master 3 does not get a grant during this address phase, but it might still have
-- a grant_s, if the access before to the slave was also done by master 3.
-- In this case the hreadyout of the slave determines the hready at master 3 during the
-- address phase.
--
-- For master 2 and master 3 an active delay_access signal during the address phase is
-- created, which means their hreadyout_wait will change to 0 in the next clock cycle.
--
-- This next clock cycle starts the data phase of the access of master 1. The slave determines
-- how long this data phase will take.
-- The grant at master 1 disappears in the data phase, as the arbiter switches because of the
-- valid address phase before. But master 1 will still be connected to the outputs of the slave
-- by grant_s, while master 2 will get the grant and will not see its hreadyout_wait anymore
-- (which will be reset to 1 by the grant). The master 3 will have neither a grant nor a
-- grant_s and will still use the signal hreadyout_wait to prolong the data phase.
--
architecture struct of ahb_multilayer_demux_hreadyout is
    signal delay_access   : std_logic_vector(g_number_of_masters-1 downto 0);
    signal hreadyout_wait : std_logic_vector(g_number_of_masters-1 downto 0);
begin
    -- An access may only be delayed in a valid address phase:
    delay_access <= hready_i and request_i and not grant_i;
    delay_access_o <= delay_access;
    process(hresetn_i, hclk_i)
    begin
        if hresetn_i='0' then
            hreadyout_wait <= (others => '1');
        elsif rising_edge(hclk_i) then
            for m in 0 to g_number_of_masters-1 loop
                if delay_access(m)='1' then
                    hreadyout_wait(m) <= '0';
                elsif grant_i(m)='1' then
                    hreadyout_wait(m) <= '1';
                end if;
            end loop;
        end if;
    end process;
    process(hreadyout_i, grant_i, grant_s_i, hreadyout_wait)
    begin
        for m in 0 to g_number_of_masters-1 loop
            if grant_i(m)='1' or -- hreadyout_i determines wether the address phase is valid.
               grant_s_i(m)='1'  -- hreadyout_i determines wether the data phase is valid.
            then
                hreadyout_o(m) <= hreadyout_i;
            else
                hreadyout_o(m) <= hreadyout_wait(m) or grant_i(m);
            end if;
        end loop;
    end process;
end architecture;
