-- Filename: ahb_slave_interface_struct.vhd
-- Created by HDL-SCHEM-Editor at Tue Dec  2 18:11:30 2025
library ieee;
use ieee.numeric_std.all;
architecture struct of ahb_slave_interface is
    constant c_master_idle : t_ahb_control(haddr(g_addr_width-1 downto 0)) := (
            haddr     => (others => '0'),
            hwrite    => '0',
            hsize     => "000",
            hburst    => (g_hburst_width-1 downto 0 => '0'),
            hprot     => (g_hprot_width-1 downto 0 => '0'),
            htrans    => "00",
            hmastlock => '0');

    -- Constants for AHB control-signals:
    constant idle   : std_logic_vector(1 downto 0) := "00";
    constant busy   : std_logic_vector(1 downto 0) := "01";
    constant nonseq : std_logic_vector(1 downto 0) := "10";
    constant seq    : std_logic_vector(1 downto 0) := "11";

    constant write  : std_logic := '1';
    constant read   : std_logic := '0';

    constant byte   : std_logic_vector(2 downto 0) := "000";
    constant half   : std_logic_vector(2 downto 0) := "001";
    constant word   : std_logic_vector(2 downto 0) := "010";

    constant single : std_logic_vector(g_hburst_width-1 downto 0) := (others => '0');

    constant lock   : std_logic := '1';
    constant nolock : std_logic := '0';

    constant hready_true : std_logic := '1';
    constant hsel_true   : std_logic := '1';
    constant hsel_false  : std_logic := '0';

    constant hprot_zero : std_logic_vector(g_hprot_width-1 downto 0) := std_logic_vector(to_unsigned(0, g_hprot_width));
    constant hprot_data : std_logic_vector(g_hprot_width-1 downto 0) := std_logic_vector(to_unsigned(1, g_hprot_width));
    constant hprot_priv : std_logic_vector(g_hprot_width-1 downto 0) := std_logic_vector(to_unsigned(2, g_hprot_width));



    signal hcontrol                : t_ahb_control(haddr(g_addr_width-1 downto 0), hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
    signal hready                  : std_logic;
    signal hready_from_other_slave : std_logic;
    signal stimulation_runs        : std_logic := '1';
begin
    -- process to drive the AHB slave interface:
    process
        procedure execute_ahb_access(         repeat    : in positive;
                                              hsel      : in std_logic;
                                              htrans    : in std_logic_vector(1 downto 0);
                                              hwrite    : in std_logic;
                                              haddr     : in std_logic_vector;
                                              hburst    : in std_logic_vector(g_hburst_width-1 downto 0);
                                              hsize     : in std_logic_vector;
                                              hprot     : in std_logic_vector(g_hprot_width-1 downto 0);
                                              hmastlock : in std_logic) is
        begin
            for i in 0 to repeat-1 loop
                hsel_o <= hsel;
                hcontrol.htrans    <= htrans;
                hcontrol.hwrite    <= hwrite;
                hcontrol.haddr     <= haddr;
                hcontrol.hsize     <= hsize;
                hcontrol.hburst    <= hburst;
                hcontrol.hprot     <= hprot;
                hcontrol.hmastlock <= hmastlock;
                wait for 1 ns; -- For a save check of hreadyout_i
                while hready='0' loop
                    wait for g_period;
                end loop;
                hwdata_o <= (others => '0') after g_period;
                hwdata_o(hcontrol.haddr'range) <= haddr after g_period;
                hwdata_o(24)                   <= hwrite after g_period;
                hwdata_o(30 downto 28)         <= hsize after g_period;
                wait for g_period - 1 ns;
            end loop;
        end procedure;
    begin
        assert g_data_width=32 or g_data_width=64 report "The testbench supports only g_data_width=32 or g_data_width=64." severity error;
        hready_from_other_slave <= '1';
        hcontrol <= c_master_idle;
        hsel_o     <= '0';
        hwdata_o   <= (others => '0');
        wait for 1 ns;
        wait for 10*g_period;
        -- At AHB write accesses the hcontrol-information (haddr, hsize, hburst, htrans) must be copied into the write data.
        -- Then the slave can check, if the correct write-data has reached the slave.
        -- It is also checked, if haddr is inside the slave address range.
        -- It is not checked, how long the master had to wait for this access.
        --
        -- 1. normal write access
        execute_ahb_access(1, hsel_true , nonseq, write, X"1232", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_false, idle  , read , X"A000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 2. write access into address gap:
        execute_ahb_access(1, hsel_true , nonseq, write, X"8123", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_false, idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 3. Three write accesses in a row, second access addresses address gap:
        execute_ahb_access(1, hsel_true , nonseq, write, X"1340", single, half , hprot_data, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"2122", single, half , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"3450", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 4. Three write accesses in a row, second access gets an additional wait state:
        execute_ahb_access(1, hsel_true , nonseq, write, X"1340", single, word , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"3421", single, byte , hprot_priv, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"1A54", single, word , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 5. Three write accesses in a row, second access causes slave error:
        execute_ahb_access(1, hsel_true , nonseq, write, X"1340", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"3456", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"1350", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 6. normal read access
        execute_ahb_access(1, hsel_true , nonseq, read , X"1230", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_false, idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 7. read access into address gap:
        execute_ahb_access(1, hsel_true , nonseq, read , X"8120", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_false, idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 8. Three read accesses in a row, second access addresses address gap:
        execute_ahb_access(1, hsel_true , nonseq, read , X"1340", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"2121", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"3452", single, byte , hprot_data, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 9. Three read accesses in a row, second access gets an additional wait state:
        execute_ahb_access(1, hsel_true , nonseq, read , X"1343", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"3421", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"1A50", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 10. Three read accesses in a row, second access causes slave error:
        execute_ahb_access(1, hsel_true , nonseq, read , X"1340", single, byte , hprot_priv, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"3456", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"1350", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        -- 11. Read and write accesses mixed:
        execute_ahb_access(1, hsel_true , nonseq, read , X"1340", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"3450", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"3456", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"8120", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, read , X"1350", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , nonseq, write, X"1230", single, byte , hprot_zero, nolock);
        execute_ahb_access(1, hsel_true , idle  , read , X"0000", single, "000", hprot_zero, nolock);
        wait for 5*g_period;
        stimulation_runs <= '0';
        wait;
    end process;
    -- process to check the read data:
    process
        procedure check_hrdata(count_length_error_messsage : in natural;
                               htrans       : in std_logic_vector(1 downto 0);
                               hwrite       : in std_logic;
                               haddr        : in std_logic_vector;
                               hsize        : in std_logic_vector;
                               hburst       : in std_logic_vector;
                               hprot        : in std_logic_vector;
                               hmastlock    : in std_logic) is
        variable slave_number : natural;
        begin
        slave_number := 12345678;
        for i in 0 to g_apb_start_addresses'length-1 loop
            if haddr(g_addr_width-1 downto g_apb_slave_address_width(i))=g_apb_start_addresses(i)(g_addr_width-1 downto g_apb_slave_address_width(i)) then
                slave_number := i;
            end if;
        end loop;
        if htrans/=idle then
            if hwrite='1' then
                report " Master: Check write access to slave " & integer'image(slave_number);
            else
                report " Master: Check read access to slave " & integer'image(slave_number);
            end if;
        end if;
        if slave_number/=12345678 then
            if hresp_i='0' then
                assert slave_number=to_integer(unsigned(hrdata_i(31 downto 28))) report "Error in master " &
                                                    " : the slave_number is wrong: the access to slave " & integer'image(slave_number) & " has a different slave number in the read data: " & integer'image(to_integer(unsigned(hrdata_i(31 downto 28)))) severity warning;
                assert haddr =hrdata_i(g_addr_width-1 downto  0) report "Error in master "  &
                                                    " : paddr is wrong: the access to address "   & to_hstring(haddr)  & " has a different address in the read data: " & to_hstring(hrdata_i(g_addr_width-1 downto  0)) severity warning;
                assert hwrite=hrdata_i(20) report "Error in master "  &
                                                    " : pwrite is wrong: the access with hwrite " & std_logic'image(hwrite) & " has a different pwrite in the read data: "   & std_logic'image(hrdata_i(20)) severity warning;
                assert g_hprot_width=0 or hprot(0)= not hrdata_i(26) report "Error in master "  &
                                                    " : pprot(2) is wrong: the access with hprot(0) " & std_logic'image(hprot(0)) & " has a different pprot(2) in the read data: "   & std_logic'image(not hrdata_i(26)) severity warning;
                assert g_hprot_width=0 or hprot(1)=     hrdata_i(24) report "Error in master "  &
                                                    " : pprot(0) is wrong: the access with hprot(1) " & std_logic'image(hprot(1)) & " has a different pprot(0) in the read data: "   & std_logic'image(hrdata_i(24)) severity warning;
                assert      '0'=     hrdata_i(25) report "Error in master "  &
                                                    " : pprot(1) is different from 0" severity warning;
            else
                assert hrdata_i=(hrdata_i'range => '0') report "Error: hrdata is different from 0 at hresp_i=1" severity error;
            end if;
        else
            assert hresp_i='1' report "Error in master: The master accessed an address gap but no error response was created" severity warning;
            assert count_length_error_messsage=1 and hresp_i='1' report "Error in master" &
                                                    ": The master accessed an address gap and the error response did not have the correct length of 2." severity warning;
        end if;
        end procedure;
        variable v_hcontrol_haddr     : std_logic_vector(g_addr_width-1 downto 0);
        variable v_hcontrol_hsize     : std_logic_vector(2 downto 0);
        variable v_hcontrol_hburst    : std_logic_vector(g_hburst_width-1 downto 0);
        variable v_hcontrol_hprot     : std_logic_vector(g_hprot_width-1 downto 0);
        variable v_hcontrol_htrans    : std_logic_vector(1 downto 0);
        variable v_hcontrol_hwrite    : std_logic;
        variable v_hcontrol_hmastlock : std_logic;
        variable count_length_error_messsage : natural;
    begin
        wait for 0.75 * g_period;
        while stimulation_runs='1' loop
            if hreadyout_i='1' and hcontrol.htrans/=idle then --valid address phase
                v_hcontrol_haddr     := hcontrol.haddr;
                v_hcontrol_hsize     := hcontrol.hsize;
                v_hcontrol_hburst    := hcontrol.hburst;
                v_hcontrol_hprot     := hcontrol.hprot;
                v_hcontrol_htrans    := hcontrol.htrans;
                v_hcontrol_hwrite    := hcontrol.hwrite;
                v_hcontrol_hmastlock := hcontrol.hmastlock;
                count_length_error_messsage := 0;
                wait for 0.5*g_period; -- Then 1/4 of the first clock period of the data phase has passed.
                while hreadyout_i='0' loop
                    if hresp_i='1' Then
                        count_length_error_messsage := count_length_error_messsage + 1;
                    end if;
                    wait for g_period;
                end loop;
                check_hrdata(count_length_error_messsage,
                             v_hcontrol_htrans, v_hcontrol_hwrite, v_hcontrol_haddr, v_hcontrol_hsize,
                             v_hcontrol_hburst, v_hcontrol_hprot, v_hcontrol_hmastlock);
                wait for 0.5*g_period; -- Then 3/4 of the clock period of the valid data phase has passed.
            else
                wait for g_period;
            end if;
        end loop;
        wait;
    end process;
    
    stimulation_runs_o <= stimulation_runs;
    hready <= hreadyout_i and hready_from_other_slave;
    hready_o <= hready;
    hcontrol_o <= hcontrol;
end architecture;
