-- Filename: ahb_apb_bridge_e.vhd
-- Created by HDL-SCHEM-Editor at Tue Dec  2 18:18:45 2025
-- The ahb_apb_bridge module converts AHB accesses into APB accesses.
-- It is both an AHB slave and an APB master for several slaves.
-- An access at APB requires at least two clock cycles.
-- However, the fastest access at AHB requires only one clock cycle.
-- Therefore, at least one wait cycle must be introduced on the AHB side.
-- Each access will therefore have at least a two-clock-cycle AHB data phase.
-- On the APB side, access starts in the first clock cycle of
-- the AHB data phase, because the AHB write data will then be valid,
-- and must be valid at APB in the first cycle of the APB access.
--
-- AHB hcontrol_i   | acc1 | acc2 | acc2 | acc3 | acc3 |
-- AHB hreadyout_o  |------|______|------|______|------|
-- AHB hwdata_i     |      | wd1  | wd1  |
-- AHB hrdata_o     |      |      | rd1  |
--
-- APB control      |      | acc1 | acc1 | acc2 |        <- PSELx, PADDR, PWRITE
-- penable_o        |      |______|------|______|
-- pready_i         |      |xxxxxx|------|xxxxxx|        <- 'x' means 0 or 1 is allowed
-- pwdata_o         |      |  wd1 | wd1  |      |               
-- prdata_i         |      |      | rd1  |      |
--
-- The APB side can introduce additional wait states:
-- AHB access       | acc1 | acc2 | acc2 | acc2 | acc3 |
-- AHB hreadyout_o  |------|______|______|------|______|
-- AHB hwdata_i     |      | wd1  | wd1  | wd1  |
-- AHB hrdata_o     |      |      |      | rd1  |
--
-- APB control      |      | acc1 | acc1 | acc1 | acc2 |
-- penable_o        |      |______|------|------|______|
-- pready_i         |      |xxxxxx|______|------|xxxxxx|
-- pwdata_o         |      | wd1  | wd1  | wd1  |      |               
-- prdata_i         |      |      |      | rd1  |      |
--
-- When the APB side asserts pslverr_i, an additional wait cycle must be introduced:
-- AHB access       | acc1 | acc2 | acc2 | acc2 | acc2 | acc3 |
-- AHB hreadyout_o  |------|______|______|______|------|______|
-- AHB hresp_o      |______|______|______|------|------|______|
-- AHB hwdata_i     |      | wd1  | wd1  | wd1  | wd1  |      |
-- AHB hrdata_o     |      |      |      |      |000000|      |
--
-- APB control      |      | acc1 | acc1 | acc1 | acc1 | acc2 |
-- penable_o        |      |______|------|------|______|______|
-- pready_i         |      |xxxxxx|______|------|xxxxxx|xxxxxx|
-- pslverr_i        |      |______|______|------|______|______|
-- pwdata_o         |      | wd1  | wd1  | wd1  | wd1  |      |        
-- prdata_i         |      |      |      | rd1  |      |      |
--
library ieee;
use ieee.std_logic_1164.all;
use work.ahb_package.all;
entity ahb_apb_bridge is
    generic (
        g_endianness              : positive := 1;
        g_number_of_apb_slaves    : positive := 2;
        g_addr_width              : positive := 32;
        g_data_width              : positive := 32;
        g_hburst_width            : natural  := 3;
        g_hprot_width             : natural  := 4;
        g_apb_slave_address_width : t_positive_list(0 to g_number_of_apb_slaves-1) := (12, 8);
        g_apb_start_addresses     : t_std_logic_vector_list(0 to g_number_of_apb_slaves-1)(g_addr_width-1 downto 0) := (X"00001000", X"00003400") 
    );
    port (
        hcontrol_i  : in  t_ahb_control  (haddr(g_addr_width-1 downto 0),hburst(g_hburst_width-1 downto 0), hprot(g_hprot_width-1 downto 0));
        hready_i    : in  std_logic;
        hsel_i      : in  std_logic;
        hwdata_i    : in  std_logic_vector (g_data_width-1 downto 0);
        pclk_i      : in  std_logic;
        prdata_i    : in  t_std_logic_vector_list (g_number_of_apb_slaves-1 downto 0)(g_data_width-1 downto 0);
        pready_i    : in  std_logic_vector (g_number_of_apb_slaves-1 downto 0);
        presetn_i   : in  std_logic;
        pslverr_i   : in  std_logic_vector (g_number_of_apb_slaves-1 downto 0);
        hrdata_o    : out std_logic_vector (g_data_width-1 downto 0);
        hreadyout_o : out std_logic;
        hresp_o     : out std_logic;
        paddr_o     : out std_logic_vector (g_addr_width-1 downto 0);
        penable_o   : out std_logic;
        pprot_o     : out std_logic_vector (2 downto 0);
        psel_o      : out std_logic_vector (g_number_of_apb_slaves-1 downto 0);
        pstrb_o     : out std_logic_vector (g_data_width/8-1 downto 0);
        pwdata_o    : out std_logic_vector (g_data_width-1 downto 0);
        pwrite_o    : out std_logic
    );
end entity ahb_apb_bridge;
