-- Filename: ahb_apb_bridge_access_handler_fsm.vhd
-- Created by HDL-FSM-Editor at Tue Dec  2 19:02:16 2025
library ieee;
use ieee.numeric_std.all;
use ieee.math_real.all;

architecture fsm of ahb_apb_bridge_access_handler is
    type t_state is (idle, setup, active, err);
    signal state : t_state;
    constant c_number_of_byte_enables : positive := g_data_width/8;
    constant c_address_offset_width   : positive := integer(log(real(c_number_of_byte_enables))/log(2.0));
    
    function create_byte_enables(address_offset : std_logic_vector(c_address_offset_width-1 downto 0);
                                 hsize          : std_logic_vector(2 downto 0)) return std_logic_vector
    is
        variable mask                : std_logic_vector(6 downto 0); -- maximum 2**7 bytes = 1024 bits
        variable byte_number_aligned : std_logic_vector(c_address_offset_width-1 downto 0);
        variable pstrb               : std_logic_vector(c_number_of_byte_enables-1 downto 0);
    begin
        case hsize is
            when "000"  => mask := "1111111";
            when "001"  => mask := "1111110";
            when "010"  => mask := "1111100";
            when "011"  => mask := "1111000";
            when "100"  => mask := "1110000";
            when "101"  => mask := "1100000";
            when "110"  => mask := "1000000";
            when "111"  => mask := "0000000";
            when others => null;
        end case;
        pstrb := (others => '0');
        for byte_number in 0 to c_number_of_byte_enables-1 loop
            if g_endianness=1 then
                byte_number_aligned := std_logic_vector(to_unsigned(byte_number,address_offset'length)) and mask(address_offset'range);
            else
                byte_number_aligned := not(std_logic_vector(to_unsigned(byte_number,address_offset'length))) and mask(address_offset'range);
            end if;
            if byte_number_aligned=address_offset then
                pstrb(byte_number) := '1';
            end if;
        end loop;
        return pstrb;
    end function;
    
    signal ahb_access               : std_logic;
    signal no_slave_selected        : std_logic;
    signal paddr                    : std_logic_vector(g_addr_width-1 downto 0);
    signal psel                     : std_logic_vector(g_number_of_apb_slaves-1 downto 0);
begin
    p_states: process (presetn_i, pclk_i)
    begin
        if presetn_i='0' then
            state <= idle;
            penable_o <= '0';
            pwrite_o  <= '0';
            pprot_o   <= "000";
            pstrb_o   <= (others => '1');
            paddr     <= (others => '0');
        elsif rising_edge(pclk_i) then
            -- Global Actions before:
            if ahb_access='1' then
                pwrite_o <= hcontrol_i.hwrite;
                paddr    <= hcontrol_i.haddr;
                if g_hprot_width/=0 then
                    pprot_o(0) <= hcontrol_i.hprot(1);
                    pprot_o(1) <= '0';
                    pprot_o(2) <= not hcontrol_i.hprot(0);
                end if;
                pstrb_o <= (others => '0');
                if c_number_of_byte_enables=1 then
                    pstrb_o(0) <= '1';
                else
                    pstrb_o <= create_byte_enables(hcontrol_i.haddr(c_address_offset_width-1 downto 0), hcontrol_i.hsize);
                end if;
            end if;
            -- State Machine:
            case state is
                when idle =>
                    if ahb_access='1' then
                        state <= setup;
                    end if;
                when setup =>
                    penable_o <= '1';
                    state <= active;
                when active =>
                    if pready_sel_i='1' or no_slave_selected='1' then
                        penable_o  <= '0';
                        if pslverr_sel_i='0' then
                            if ahb_access='1' then
                                state <= setup;
                            else
                                state <= idle;
                            end if;
                        else
                            state <= err;
                        end if;
                    end if;
                when err =>
                    if ahb_access='1' then
                        state <= setup;
                    else
                        state <= idle;
                    end if;
            end case;
            -- Global Actions after:
            
        end if;
    end process;
    p_state_actions: process (no_slave_selected, psel, pready_sel_i, pslverr_sel_i, state)
    begin
        -- State Actions:
        case state is
            when idle=>
                psel_o      <= (others => '0');
                hreadyout_o <= '1';
                hresp_o     <= '0';
            when setup=>
                psel_o      <= psel;
                hreadyout_o <= '0';
                hresp_o     <= no_slave_selected;
            when active=>
                psel_o <= psel;
                if no_slave_selected='1' then
                    hreadyout_o <= '1';
                else
                    hreadyout_o <= pready_sel_i and not pslverr_sel_i;
                end if;
                hresp_o <= no_slave_selected or pslverr_sel_i;
            when err=>
                psel_o      <= (others => '0');
                hreadyout_o <= '1';
                hresp_o     <= '1';
        end case;
    end process;
    -- Global Actions combinatorial:
    -- Check for an AHB access:
    ahb_access <= '1' when hready_i            ='1' and
                           hsel_i              ='1' and
                           hcontrol_i.htrans(1)='1' else '0'; -- NONSEQ, SEQ
    -- Create PSELx signals:
    process(paddr)
    begin
        psel <= (others => '0');
        for i in g_number_of_apb_slaves-1 downto 0 loop
            if paddr                   (g_addr_width-1 downto g_apb_slave_address_width(i))=
               g_apb_start_addresses(i)(g_addr_width-1 downto g_apb_slave_address_width(i))
            then
                psel(i) <= '1';
            end if;
        end loop;
    end process;
    -- Check for an access into a address gap:
    no_slave_selected <= '1' when psel=(g_number_of_apb_slaves-1 downto 0 => '0') else '0';
    -- To APB-slave:
    paddr_o  <= paddr;
end architecture;
